/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.merchandise.domain.impl;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.merchandise.dao.OpChannelMapper;
import com.thebeastshop.pegasus.merchandise.domain.BaseDomain;
import com.thebeastshop.pegasus.merchandise.domain.ChannelSelector;
import com.thebeastshop.pegasus.merchandise.domain.OpChannelDomain;
import com.thebeastshop.pegasus.merchandise.domain.OpChnCanSaleProdDomain;
import com.thebeastshop.pegasus.merchandise.exception.OperationException;
import com.thebeastshop.pegasus.merchandise.exception.OperationExceptionErrorCode;
import com.thebeastshop.pegasus.merchandise.model.OpChannel;
import com.thebeastshop.pegasus.merchandise.model.OpChannelExample;
import com.thebeastshop.pegasus.merchandise.model.OpChnCanSaleProd;
import com.thebeastshop.pegasus.merchandise.cond.OpChannelCond;
import com.thebeastshop.pegasus.merchandise.vo.OpChannelVO;
import com.thebeastshop.pegasus.util.comm.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @Author: LK【kai.li@thebeastshop.com】
 * @Date: 2016/9/5 13:27
 */
@Component("opChannelDomain")
public class OpChannelDomainImpl extends AbstractBaseDomain<OpChannelVO, OpChannel> implements OpChannelDomain {


    @Autowired
    private OpChannelMapper opChannelMapper;
    @Autowired
    private OpChnCanSaleProdDomain opChnCanSaleProdDomain;

    /**
     * 根据条件查找渠道
     *
     * @param cond
     * @return 渠道列表
     */
    @Override
    public List<OpChannel> findChannelByCond(final String cond) {
        final OpChannelExample example = new OpChannelExample();
        example.createCriteria().andCodeLike(SQLUtils.allLike(cond));
        example.or().andNameLike(SQLUtils.allLike(cond));
        example.setOrderByClause("CODE limit 20");
        return opChannelMapper.selectByExample(example);
    }

    @Override
    public boolean create(final OpChannel channel) {
        channel.setCreateTime(DateUtil.getNow());
        try {
            opChannelMapper.insert(channel);
        } catch (final Exception e) {
            e.printStackTrace();
            throw new OperationException(OperationExceptionErrorCode.CHANNEL_CREATE_ERROR, "渠道新增出错！");
        }
        final Map<String, Object> params = new HashMap<>();
        params.put("id", channel.getId());
        params.put("type", channel.getChannelType());
        final String code = CodeGenerator.getInstance().generate("OP_CHANNEL_CODE", params);
        channel.setCode(code);
        validateCode(channel.getCode(), channel.getId());
        return update(channel);
    }

    @Override
    public boolean update(final OpChannel channel) {
        return opChannelMapper.updateByPrimaryKeySelective(channel) != 0;
    }

    @Override
    public boolean deleteById(final Long id) {
        validateDelete(id);
        return opChannelMapper.deleteByPrimaryKey(id) != 0;
    }

    @Override
    public OpChannel findById(final Long id) {
        return opChannelMapper.selectByPrimaryKey(id);
    }

    /**
     * 条件查询 渠道
     *
     * @param cond
     * @return
     */
    @Override
    public List<OpChannel> findByCriteria(final OpChannelCond cond) {
        final OpChannelExample example = new OpChannelExample();
        final OpChannelExample.Criteria criteria = example.createCriteria();
        final String name = cond.getName();
        if (StringUtils.isNotBlank(name)) {
            criteria.andNameLike(SQLUtils.allLike(name));
        }
        final String code = cond.getCode();
        if (StringUtils.isNotBlank(code)) {
            criteria.andCodeLike(SQLUtils.allLike(code));
        }
        example.setOrderByClause(" ID " + cond.getCriteriaStr());
        final int channelType = (int) NumberUtil.avoidNull(cond.getChannelType(), -1);
        if (channelType > -1) {
            criteria.andChannelTypeEqualTo(channelType);
        }

        return opChannelMapper.selectByExample(example);
    }

    /**
     * 根据编码查找
     *
     * @param code
     * @return
     */
    @Override
    public OpChannel findByCode(final String code) {
        final OpChannelExample example = new OpChannelExample();
        final OpChannelExample.Criteria criteria = example.createCriteria();
        criteria.andCodeEqualTo(code);
        final List<OpChannel> channels = opChannelMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(channels)) {
            return null;
        }
        return channels.get(0);
    }

    @Override
    public List<OpChannel> findAll() {
        final OpChannelExample example = new OpChannelExample();
        //roys 待测试是否影响
        final OpChannelExample.Criteria criteria = example.createCriteria();
        final List<OpChannel> channels = opChannelMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(channels)) {
            return Collections.emptyList();
        } else {
            return channels;
        }
    }

    private void validateCode(final String code, final long id) {
        final OpChannel channel = findByCode(code);
        if (channel != null && channel.getId() != id) {
            throw new OperationException(OperationExceptionErrorCode.DUPLICATE_CHANNEL_CODE, "渠道编码已经存在，请重新输入！");
        }
    }

    private void validateDelete(final Long id) {
        final OpChannel channel = findById(id);

        List<OpChnCanSaleProd> skus = null;
        try {
            skus = opChnCanSaleProdDomain.findByChannelCode(channel.getCode());
        } catch (final Exception e) {
            e.printStackTrace();
        }
        if (CollectionUtils.isNotEmpty(skus)) {
            throw new OperationException(OperationExceptionErrorCode.CHANNEL_DELETE_CANSALESKU_ERROR, "渠道有可售SKU，不能删除！");
        }
    }

    @Override
    public List<OpChannel> selectAllFlowerShops() {
        return opChannelMapper.selectAllFlowerShops();
    }

    @Override
    public List<OpChannel> findChannelByType(final int type) {
        return opChannelMapper.selectChannelByType(type);
    }

    @Override
    public List<ChannelSelector> findByChannelType(Integer type) {
        OpChannelExample example = new OpChannelExample();
        if(null != type) {
            example.createCriteria().andChannelTypeEqualTo(type);
    }
        List<OpChannel> list = opChannelMapper.selectByExample(example);
        return convert2ChannelSelectorList(list);
    }

    /**
     * convert OpChannel to ChannelSelector
     * @return
     */
    public static ChannelSelector convert2ChannleSelector(OpChannel opChannel){
        ChannelSelectorImpl domain = new ChannelSelectorImpl();
        domain.setCode(opChannel.getCode());
        domain.setName(opChannel.getName());
        return domain;
    }

    /**
     * convert OpChannel集合 to ChannelSelector集合
     *
     * @param entities 待转换集合
     *
     * @return
     */
    public static List<ChannelSelector> convert2ChannelSelectorList(List<OpChannel> entities) {
        if(CollectionUtils.isEmpty(entities)) {
            return null;
        }
        List<ChannelSelector> list = Lists.newArrayList();
        for (OpChannel entity : entities) {
            list.add(convert2ChannleSelector(entity));
        }
        return list;
    }
}