package com.thebeastshop.pegasus.merchandise.service.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.domain.CampaignDomain;
import com.thebeastshop.pegasus.merchandise.service.McOpSpvService;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.vo.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Maps;
import com.thebeastshop.pegasus.merchandise.client.SpvESClient;
import com.thebeastshop.pegasus.merchandise.cond.PsSpvCond;
import com.thebeastshop.pegasus.merchandise.constants.SearchType;
import com.thebeastshop.pegasus.merchandise.domain.PsSkuDomain;
import com.thebeastshop.pegasus.merchandise.service.McPsSpvService;
import com.thebeastshop.pegasus.merchandise.util.McReflectionUtil;

/**
 * @author gongjun
 * @since 2016-11-03
 */
@Service("mcPsSpvService")
public class McPsSpvServiceImpl implements McPsSpvService {

    @Autowired
    private SpvESClient client;

    @Autowired
    private PsSkuDomain psSkuDomain;
    

    @Autowired
    private CampaignDomain campaignDomain;

    @Autowired
    private McPsProductService mcPsProductService;

    @Autowired
    private McOpSpvService mcOpSpvService;
    
	/**
	 * 完善数据
	 * 
	 * @param spvs
	 */
	private List<PsSpvVO> fillingData(List<PsSpvVO> spvs) {
		if (CollectionUtils.isNotEmpty(spvs)) {
			// 目前只需要对单品做处理
			List<String> skuCodes = Lists.newArrayList();
			List<String> productCodes = Lists.newArrayList();
			for (PsSpvVO spvVO : spvs) {
				if (spvVO.getCombined() == null || !spvVO.getCombined()) {
					skuCodes.add(spvVO.getSkuCode());
					productCodes.add(spvVO.getProductCode());
				}
			}
			if (CollectionUtils.isNotEmpty(productCodes) && CollectionUtils.isNotEmpty(skuCodes)) {
				Map<String, PsSkuVO> mapBySkuCodes = psSkuDomain.mapBySkuCodes(skuCodes);
				Map<String, PsProductVO> mapByProdCodes = mcPsProductService.mapByProdCodes(productCodes);
				Iterator<PsSpvVO> iterator = spvs.iterator();
				while(iterator.hasNext()){
					PsSpvVO spvVO = iterator.next();
					if (spvVO != null && (spvVO.getCombined() == null || !spvVO.getCombined())) {
						PsSkuVO sku = mapBySkuCodes.get(spvVO.getSkuCode());
						PsProductVO product = mapByProdCodes.get(spvVO.getProductCode());
						if (sku != null && product != null) {
							// 补全spv销售价格
							spvVO.setSalesPrice(BigDecimal.valueOf(sku.getSalesPrice()));
							// 补全spv规格属性
							if (StringUtils.isNotBlank(product.getSkuCategoryDesc())
									&& StringUtils.isNotBlank(sku.getSkuCateName())) {
								PsAttributeValuePairVO e = new PsAttributeValuePairVO();
								e.setAttrName(product.getSkuCategoryDesc());
								e.setValue(sku.getSkuCateName());
								spvVO.setAttributeValuePairList(Lists.newArrayList(e));
							}
						} else {
							// spv的product或sku不存在则删除spv
							iterator.remove();
						}
					}
				}
			}
		}
		return spvs;
	}
    
    // ---------------------------------------------

    @Override
    public List<PsSpvVO> findByProdId(Long prodId) {
        return fillingData(client.findByField("productId", prodId));
    }

    @Override
    public PsSpvVO findSpvBySkuCode(String prodCode, String skuCode) {
        Map<String, Object> params = new HashMap<>();
        params.put("prodCode", QueryBuilders.queryStringQuery(prodCode).field("prodCode"));
        params.put("skuCode", QueryBuilders.queryStringQuery(skuCode).field("skuCode"));
        List<PsSpvVO> spvList = fillingData(client.findByFields(params, SearchType.AND));
        if (CollectionUtils.isEmpty(spvList)) return null;
        //新增IPOS，取出数据库也存在的SPV
        for(PsSpvVO spv:spvList){
            Boolean flag = mcOpSpvService.getSpvById(spv.getSpvId());
            if(flag){
                return spv;
            }
        }
        return spvList.get(0);
    }

    @Override
    public List<PsSpvVO> findWithSkuByProdId(Long prodId) {
        List<PsSpvVO> spvList = findByProdId(prodId);
        if (CollectionUtils.isNotEmpty(spvList)) {
            List<String> skuCodes = new ArrayList<>();
            Map<String, PsSkuCombinationVO> skuCombinationVOMap = new HashMap<>();
            Map<String, PsSpvVO> spvVOMap = new HashMap<>();
            for (PsSpvVO spvVO : spvList) {
                if (spvVO.getCombined() != null && spvVO.getCombined()) {
                    // 如果是组合商品
                    List<PsSkuCombinationVO> skuCombinationList = spvVO.getSkuCombinationList();
                    if (CollectionUtils.isNotEmpty(skuCombinationList)) {
                        for (PsSkuCombinationVO skuCombinationVO : skuCombinationList) {
                            skuCodes.add(skuCombinationVO.getSubSkuCode());
                            skuCombinationVOMap.put(skuCombinationVO.getSubSkuCode(), skuCombinationVO);
                        }
                    }
                }
                else {
                    // 如果是单品
                    skuCodes.add(spvVO.getSkuCode());
                    spvVOMap.put(spvVO.getSkuCode(), spvVO);
                }
            }

            List<PsSkuVO> skuVOs = psSkuDomain.findBySkuCodes(skuCodes);
            if (CollectionUtils.isNotEmpty(skuVOs)) {
                for (PsSkuVO skuVO : skuVOs) {
                    PsSpvVO spvVO = spvVOMap.get(skuVO.getSkuCode());
                    if (spvVO != null) {
                        spvVO.setSkuVO(skuVO);
                    }
                    else {
                        PsSkuCombinationVO skuCombinationVO = skuCombinationVOMap.get(skuVO.getSkuCode());
                        if (skuCombinationVO != null) {
                            skuCombinationVO.setSubSkuVO(skuVO);
                        }
                    }
                }
            }
        }
        return fillingData(spvList);
    }

    @Override
    public PsSpvVO findBySpvId(Long spvId) {
        List<PsSpvVO> list = fillingData(client.findByField(client.getPrimaryKeyName(), spvId));
        if (CollectionUtils.isEmpty(list))
            return null;
        return list.get(0);
    }

    @Override
    public PsSpvVO findWithSkuBySpvId(Long spvId) {
        PsSpvVO spvVO = findBySpvId(spvId);
        if (spvId == null) return null;
        if (spvVO.getCombined() != null && spvVO.getCombined()) {
            // 如果是组合商品
            List<String> skuCodes = new ArrayList<>();
            List<PsSkuCombinationVO> skuCombinationList = spvVO.getSkuCombinationList();
            Map<String, PsSkuCombinationVO> skuCombinationVOMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(skuCombinationList)) {
                for (PsSkuCombinationVO skuCombinationVO : skuCombinationList) {
                    skuCodes.add(skuCombinationVO.getSubSkuCode());
                    skuCombinationVOMap.put(skuCombinationVO.getSubSkuCode(), skuCombinationVO);
                }
            }
            List<PsSkuVO> skuVOs = psSkuDomain.findBySkuCodes(skuCodes);
            if (skuVOs != null) {
                for (PsSkuVO skuVO : skuVOs) {
                    PsSkuCombinationVO skuCombinationVO = skuCombinationVOMap.get(skuVO.getSkuCode());
                    if (skuCombinationVO != null) {
                        skuCombinationVO.setSubSkuVO(skuVO);
                    }
                }
            }
        }
        else {
            // 如果是单品
            PsSkuVO skuVO = psSkuDomain.findBySkuCode(spvVO.getSkuCode());
            spvVO.setSkuVO(skuVO);
        }
        return spvVO;
    }

    @Override
	public List<PsSpvVO> findBySpvIds(List<Long> spvIds) {
		if (CollectionUtils.isNotEmpty(spvIds)) {
            Map<String, Object> params = new HashMap<>();
            params.put(client.getPrimaryKeyName(), spvIds);
            params.put("pagenum", 9999);
            return fillingData(client.findByFields(params, SearchType.OR));
		}
		return null;
	}
    
    @Override
	public Map<Long, PsSpvVO> mapBySpvIds(List<Long> spvIds) {
		if (CollectionUtils.isNotEmpty(spvIds)) {
			Map<Long, PsSpvVO> rt = Maps.newHashMapWithExpectedSize(spvIds.size());
			List<PsSpvVO> spvs = findBySpvIds(spvIds);
			for (PsSpvVO spv : spvs) {
				rt.put(spv.getSpvId(), spv);
			}
			return rt;
		}
		return null;
	}

    @Override
    public List<PsSpvVO> findByCond(PsSpvCond cond) {
        Map<String, Object> params = McReflectionUtil.getFieldNameValue(cond);
        return fillingData(client.findByFields(params, SearchType.ANDLIKE));
    }

    @Override
    public List<PsSpvVO> findBySpvIds(List<Long> spvIds, PsMatchCampaignCond cond) {
        List<PsSpvVO> result = Lists.newArrayList();
        if(CollectionUtils.isEmpty(spvIds)) {
            return result;
        }
        List<PsSpvVO> spvList = findBySpvIds(spvIds);
        if(CollectionUtils.isEmpty(spvList)) {
            return result;
        }
        List<PriceCalResultVO> results;
        PsProductVO productVO;
        for (PsSpvVO spvVO : spvList) {
            productVO = mcPsProductService.findProdBySpvId(spvVO.getSpvId());
            if(productVO == null) {
                continue;
            }
            // 匹配Product维度活动
            campaignDomain.matchCampaigns(productVO, cond);
            // 匹配SPV维度活动
            campaignDomain.matchCampaigns(spvVO, cond, productVO.getCrossBorderFlag());
            // 计算价格
            results = Lists.newArrayList();
            mcPsProductService.calculateDiscountPrice(productVO, spvVO, cond, results);
            result.add(spvVO);
        }
        return result;
    }
}
