package com.thebeastshop.pegasus.merchandise.service;

import com.thebeastshop.pegasus.merchandise.IService.IProdDiscountService;
import com.thebeastshop.pegasus.merchandise.cond.OpProdDiscountCond;
import com.thebeastshop.pegasus.merchandise.dao.OpProdDiscountMapper;
import com.thebeastshop.pegasus.merchandise.dao.OpProductMapper;
import com.thebeastshop.pegasus.merchandise.domain.OpProductDomain;
import com.thebeastshop.pegasus.merchandise.model.OpProdDiscount;
import com.thebeastshop.pegasus.merchandise.model.OpProduct;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuVO;
import com.thebeastshop.pegasus.merchandise.vo.ProdDiscountSkuVO;
import com.thebeastshop.pegasus.merchandise.vo.ProdDiscountVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;


import java.util.*;

/**
 * Created by admin on 2017/8/16.
 */
@Service("prodDiscountService")
public class ProdDiscountService implements IProdDiscountService {

    private static final Logger logger = LoggerFactory.getLogger(ProdDiscountService.class);

    @Autowired
    private OpProdDiscountMapper mapper;

    @Autowired
    private OpProductMapper productMapper;

    @Autowired
    private OpProductDomain opProductDomain;

    private final static String MEMBER_LEVEL_ID = "2,3";  //对应会员等级中的老虎大象
    private final static Float DISCOUNT = 0.88f;

    @Override
    public int addByProdId(Long prodId, Long operatorId, String operatorName) {
        if(prodId==null) throw new RuntimeException("prodId must be not null");
        if(operatorId==null) throw new RuntimeException("operatorName must be not null");
        if(operatorName==null) throw new RuntimeException("operatorName must be not null");

        OpProdDiscount record = new OpProdDiscount();
        record.setProductId(prodId);
        record.setMemberLevelId(MEMBER_LEVEL_ID);
        record.setDiscount(DISCOUNT);
        record.setOperatorId(operatorId);
        record.setOperatorName(operatorName);
        record.setOperateTime(new Date());
        record.setActive(1);
        return mapper.insert(record);
    }

    @Override
    public int batchAddByProdId(List<Long> prodIds, Long operatorId, String operatorName) {
        if(CollectionUtils.isEmpty(prodIds)) throw new RuntimeException("prodIds must be not null");
        if (operatorId==null) throw new RuntimeException("operatorId must be not null");
        if(StringUtils.isBlank(operatorName)) throw new RuntimeException("operatorNam must be not null");

        List<OpProdDiscount> records = new ArrayList<>();
        for(Long prodId : prodIds){
            OpProdDiscount record = new OpProdDiscount();
            record.setProductId(prodId);  //将商品id赋值给
            record.setMemberLevelId(MEMBER_LEVEL_ID);//老虎会员以上，即老虎大象会员
            record.setDiscount(0.88f);//88折扣
            record.setOperatorId(operatorId);
            record.setOperatorName(operatorName);
            record.setOperateTime(new Date());
            record.setActive(1);
            records.add(record);
        }
        return mapper.batchInsert(records);
    }

    @Override
    public int add(String code,Long operatorId,String operatorName){
        if(StringUtils.isBlank(code)) throw new RuntimeException("code must be not null");
        if(operatorId==null) throw new RuntimeException("operatorId must be not null");
        if(operatorName==null) throw new RuntimeException("operatorName must be not null");

        OpProduct product = productMapper.selectByCode(code);
        OpProdDiscount record = new OpProdDiscount();
        record.setProductId(product.getId());
        record.setMemberLevelId(MEMBER_LEVEL_ID);
        record.setDiscount(DISCOUNT);
        record.setOperatorId(operatorId);
        record.setOperatorName(operatorName);
        record.setOperateTime(new Date());
        record.setActive(1);

        return mapper.insert(record);
    }

    @Override
    public int batchAdd(List<String> codes,Long operatorId,String operatorName) {
        if(CollectionUtils.isEmpty(codes)) throw new RuntimeException("codes must be not null");
        if(operatorId==null) throw new RuntimeException("operatorId must be not null");
        if(operatorName==null) throw new RuntimeException("operatorName must be not null");

        List<OpProdDiscount> records = new ArrayList<>();
        List<OpProduct> products;
        try {
            products = mapper.selectProdByCodes(codes);
        }catch (Exception e){
            logger.error("batchAdd-->selectProdByCodes错误{}",e);
            throw e;
        }

        Date now = new Date();
        for(OpProduct product:products){
            OpProdDiscount record = new OpProdDiscount();
            record.setProductId(product.getId());  //将商品id赋值给
            record.setMemberLevelId(MEMBER_LEVEL_ID);//老虎会员以上，即老虎大象会员
            record.setDiscount(0.88f);//88折扣
            record.setOperatorId(operatorId);
            record.setOperatorName(operatorName);
            record.setOperateTime(now);
            record.setActive(1);
            records.add(record);
        }

        return CollectionUtils.isNotEmpty(records)? mapper.batchInsert(records) : -1;
    }

    @Override
    public boolean checkProdCode(String code) {
        if(StringUtils.isBlank(code)) throw new RuntimeException("code must be not null");

        return productMapper.selectByCode(code) != null? true : false;
    }

    @Override
    public List<String> checkProdCodes(List<String> codes) {
        if(CollectionUtils.isEmpty(codes)) throw new RuntimeException("codes must be not null");

        return mapper.checkProdByCodes(codes);
    }

    @Override
    public List<Long> checkExistProdIds(List<Long> productIds) {
        if(CollectionUtils.isEmpty(productIds)) throw new RuntimeException("productIds must be not null");

        return mapper.checkExistProdIds(productIds);
    }

    @Override
    public List<ProdDiscountVO> queryProdByCodes(List<String> codes) {
        if(CollectionUtils.isEmpty(codes)) throw new RuntimeException("codes must be not null");

        List<Map<String,Object>> prodList;
        try {
            prodList = mapper.queryProdByCodes(codes);
        }catch (Exception e){
            logger.error("queryProdByCodes-->queryProdByCodes错误{}",e);
            throw e;
        }

        List<ProdDiscountVO> vos = new ArrayList<>();
        for (Map<String,Object> prod : prodList){
            ProdDiscountVO vo = new ProdDiscountVO();
            vo.setProdId(((Integer)prod.get("prodId")).longValue());
            vo.setProdName((String)prod.get("prodName"));
            String prodCode = (String)prod.get("prodCode");
            vo.setProdCode(prodCode);
            vo.setSkuVOs(getSkuList(prodCode));
            vos.add(vo);
        }
        return vos;
    }

    @Override
    public int deleteByProdCode(String code) {
        if(StringUtils.isBlank(code)) throw new RuntimeException("code must be not null");

        return mapper.deleteByProCode(code);
    }

    @Override
    public int deleteByProdCodes(List<String> codes) {
        if(CollectionUtils.isEmpty(codes)) throw new RuntimeException("codes must be not null");

        return mapper.batchDeleteByProCodes(codes);
    }

    @Override
    public List<ProdDiscountVO> queryByCond(OpProdDiscountCond cond) {
        List<ProdDiscountVO> vos;
        try {
            vos = mapper.queryByCond(cond);
        }catch (Exception e){
            logger.error("queryByCond-->queryByCond出错,{}",e);
            throw e;
        }
        for(ProdDiscountVO vo : vos){
            vo.setSkuVOs(getSkuList(vo.getProdCode()));
        }
        return vos;
    }

    @Override
    public int deleteById(Long id) {
        if(id==null) throw new RuntimeException("id must be not null");

        return mapper.deleteById(id);
    }

    @Override
    public int deleteByIds(List<Long> ids) {
        if(CollectionUtils.isEmpty(ids)) throw new RuntimeException("ids must be not null");

        return  mapper.deleteByIds(ids);
    }

    @Override
    public List<ProdDiscountVO> queryByIds(List<Long> ids) {
        if(CollectionUtils.isEmpty(ids)) throw new RuntimeException("ids must be not null");

        List<ProdDiscountVO> vos = mapper.queryProdByIds(ids);
        for(ProdDiscountVO vo : vos){
            String prodCode = vo.getProdCode();
            vo.setSkuVOs(getSkuList(prodCode));
        }
        return vos;
    }

    @Override
    public List<ProdDiscountVO> queryAllProd() {
        List<ProdDiscountVO> skuList = mapper.queryAllProd();
        return skuList;
    }

    private List<ProdDiscountSkuVO> getSkuList(String prodCode){
        List<PcsSkuVO> opSkus = new LinkedList<>();
        try{
            opSkus = opProductDomain.getSkuListByProductCode(prodCode);
        }catch(Exception e){
            logger.error("getSkuList-->getSkuListByProductCode错误{}",e);
        }
        List<ProdDiscountSkuVO> skuList = new LinkedList<>();
        for(PcsSkuVO opSku : opSkus){
            ProdDiscountSkuVO sku = new ProdDiscountSkuVO();
            sku.setSkuCode(opSku.getCode());
            sku.setSkuName(opSku.getNameCn());
            skuList.add(sku);
        }
        return skuList;
    }

}

