package com.thebeastshop.pegasus.merchandise.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.elasticsearch.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import com.thebeastshop.cart.resp.CartProductPack;
import com.thebeastshop.cart.service.CartService;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.pegasus.merchandise.IService.IInterestGoodsService;
import com.thebeastshop.pegasus.merchandise.exception.InterestException;
import com.thebeastshop.pegasus.merchandise.exception.InterestExceptionEnum;
import com.thebeastshop.pegasus.merchandise.util.DateUtils;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.scm.cond.InterestGoodsCond;
import com.thebeastshop.scm.dao.InterestGoodsDao;
import com.thebeastshop.scm.dao.InterestGoodsProvideDao;
import com.thebeastshop.scm.dao.InterestGoodsSkuDao;
import com.thebeastshop.scm.po.InterestGoods;
import com.thebeastshop.scm.po.InterestGoodsProvide;
import com.thebeastshop.scm.po.InterestGoodsSku;
import com.thebeastshop.scm.vo.interest.InterestGoodsGetVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsProvideVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsSaveVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsSkuVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsVO;
import com.thebeastshop.support.enums.CartPackSource;

/**
 * 商品权益Service
 * @Class Name InterestGoodsService
 * @Author Xiang.Gao
 * @Create In 2017-5-9
 */
@Service
public class InterestGoodsService implements IInterestGoodsService {
	
	private static final Logger logger = LoggerFactory.getLogger(InterestGoodsService.class);

	@Resource 
	private InterestGoodsDao interestGoodsDao;
	
	@Resource 
	private InterestGoodsSkuDao interestGoodsSkuDao;
	
	@Resource 
	private InterestGoodsProvideDao interestGoodsProvideDao;
	
	@Resource
	private CartService cartService;
	
	@Resource
	McPsProductService mcPsProductService;
	
	@SuppressWarnings("rawtypes")
	@Override
	public List<Map> list(InterestGoodsCond cond) {
		return interestGoodsDao.list(cond);
	}
	
	@Override
	public InterestGoods selectInterestById(Integer interestGoodsId) {
		return interestGoodsDao.selectOne(interestGoodsId);
	}
	
	@Override
	public InterestGoodsVO selectInterestGoodsSku(Integer interestGoodsId) {
		InterestGoods interestGoods = interestGoodsDao.selectOne(interestGoodsId);
		List<InterestGoodsSku> interestGoodsSkuList = interestGoodsSkuDao.byInterestGoodsId(interestGoodsId);
		return new InterestGoodsVO(interestGoods, interestGoodsSkuList);
	}
	
	@Override
	public int sumQuantityByInterestId(Integer interestGoodsId) {
		return interestGoodsSkuDao.sumQuantityByInterestId(interestGoodsId);
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public Integer save(InterestGoodsSaveVO vo) {
		InterestGoods interestGoods = vo.toInterestGoods();
		int res = interestGoodsDao.insertOrUpdate(interestGoods);
		if (res != 1) {
			throw new InterestException(InterestExceptionEnum.ERROR_SAVE_INTEREST);
		}
		
		List<InterestGoodsSku> interestGoodsSkuList = vo.toInterestGoodsSku(interestGoods.getId());
		for (InterestGoodsSku interestGoodsSku : interestGoodsSkuList) {
			res = interestGoodsSkuDao.insertOrUpdate(interestGoodsSku);
			if (res != 1) {
				throw new InterestException(InterestExceptionEnum.ERROR_SAVE_INTEREST_SKU);
			}
		}
		return interestGoods.getId();
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public void delete(int interestGoodsId) {
		int res = interestGoodsDao.delete(interestGoodsId);
		if(res == 0) {
			throw new InterestException(InterestExceptionEnum.ERROR_DELETE_INTEREST);
		}
		res = interestGoodsSkuDao.deleteByInterestGoodsId(interestGoodsId);
		if(res == 0) {
			throw new InterestException(InterestExceptionEnum.ERROR_DELETE_INTEREST_SKU);
		}
	}
	
	@Override
	public List<InterestGoodsProvideVO> getProvideRecord(InterestGoodsCond cond) {
		// TODO Auto-generated method stub
		return interestGoodsProvideDao.queryProvideRecord(cond);
	}
	
	
	/**-----------------------------------   以下接口提供给外部调用       --------------------------*/
	
	@Transactional("scmDataSourceTx")
	@Override
	public InterestGoodsGetVO getInterestGoods(Integer memberId, Integer interestGoodsId){
		logger.info("getInterestGoods param member:{},interestGoodsId:{}", memberId, interestGoodsId);
		
		if(memberId == null || interestGoodsId == null) {
			throw new InterestException(InterestExceptionEnum.ERROR_PARAM_MISS);
		}
		InterestGoods interestGoods = interestGoodsDao.selectOne(interestGoodsId);
		if (interestGoods == null) {
			logger.error(InterestExceptionEnum.ERROR_NOT_FOUND.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND);
		}
		// 权益领取时间校验
		if (interestGoods.getReceiptStartTime().after(DateUtils.getCurrentDate()) 
				|| interestGoods.getReceiptEndTime().before(DateUtils.getCurrentDate())) {
			throw new InterestException(InterestExceptionEnum.ERROR_RECEIVE_SPOCE);
		}
		// 乐观锁发放权益
		InterestGoodsSku interestGoodsSku = interestGoodsSkuDao.getInStockByInterestGoodsId(interestGoodsId);
		if (interestGoodsSku == null) {
			logger.error(InterestExceptionEnum.ERROR_SKU_NO_STOCK.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_SKU_NO_STOCK);
		}
		// 修改权益领取数量
		boolean isSuccess = interestGoodsSkuDao.addReceiveQuantity(interestGoodsSku.getId());
		if(!isSuccess) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_RECEIVE.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_RECEIVE);
		}
		// 保存发放记录
		int res = interestGoodsProvideDao.save(memberId, interestGoodsId, interestGoodsSku.getId());
		if(res != 1) {
			logger.error(InterestExceptionEnum.ERROR_PROVIDE_INTEREST.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_PROVIDE_INTEREST);
		}
		InterestGoodsGetVO interestGoodsGetVO = new InterestGoodsGetVO(interestGoods, interestGoodsSku);
		logger.info("getInterestGoods success:{}", interestGoodsGetVO.toString());
		return interestGoodsGetVO;
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public void cash(Integer memberId, List<InterestGoodsSkuVO> interestGoodsSkuVOList) {
		logger.info("interestGoods cash member:{},skuList:{}", memberId, interestGoodsSkuVOList.toString());
		
		if(ObjectUtils.isEmpty(memberId) || CollectionUtils.isEmpty(interestGoodsSkuVOList)) {
			throw new InterestException(InterestExceptionEnum.ERROR_PARAM_NULL);
		}
		// 权益兑取验证
		InterestGoods interestGoods = null;
		for(InterestGoodsSkuVO interestGoodsSkuVO : interestGoodsSkuVOList) {
			if (ObjectUtils.isEmpty(interestGoodsSkuVO.getCashQuantity()) 
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getInterestGoodsId())
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getSkuCode())) {
				throw new InterestException(InterestExceptionEnum.ERROR_PARAM_MISS);
			}
			Integer interestGoodsId = interestGoodsSkuVO.getInterestGoodsId();
			String skuCode = interestGoodsSkuVO.getSkuCode();
			InterestGoodsSku interestGoodsSku = interestGoodsSkuDao.getInterestGoodsSku(interestGoodsId, skuCode);
			if(ObjectUtils.isEmpty(interestGoodsSku)) {
				throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND);
			}
			interestGoods = interestGoodsDao.selectOne(interestGoodsId);
			if(interestGoods.getCashStartTime().after(DateUtils.getCurrentDate()) 
					|| interestGoods.getCashEndTime().before(DateUtils.getCurrentDate())) {
				throw new InterestException(InterestExceptionEnum.ERROR_CASH_SPOCE);
			}
			int unCashQuantity = interestGoodsProvideDao.countCashQuantity(
					memberId, interestGoodsId, interestGoodsSku.getId() , true);
			if(interestGoodsSkuVO.getCashQuantity() > unCashQuantity) {
				throw new InterestException(InterestExceptionEnum.ERROR_CASH_EXCESS);
			}
			interestGoodsSkuVO.setInterestGoodsSkuId(interestGoodsSku.getId());
		}
		//  验证通过更新权益兑取数量
		if(!interestGoodsSkuDao.updateCash(interestGoodsSkuVOList)) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_EXCESS.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_EXCESS);
		}
		// 验证通过更新用户权益兑取状态
		if(!interestGoodsProvideDao.updateCash(memberId, interestGoodsSkuVOList)) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_STATUS.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_STATUS);
		}
		logger.info("member:{} cash success ",memberId);
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public void back(Integer memberId, List<InterestGoodsSkuVO> interestGoodsSkuVOList) {
		logger.info("back interestGoods member:{}, skuList:{}", memberId, interestGoodsSkuVOList.toString());
		
		if(ObjectUtils.isEmpty(memberId) || CollectionUtils.isEmpty(interestGoodsSkuVOList)) {
			throw new InterestException(InterestExceptionEnum.ERROR_PARAM_NULL);
		}
		for(InterestGoodsSkuVO interestGoodsSkuVO : interestGoodsSkuVOList) {
			if (ObjectUtils.isEmpty(interestGoodsSkuVO.getCashQuantity()) 
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getInterestGoodsId())
					|| ObjectUtils.isEmpty(interestGoodsSkuVO.getSkuCode())) {
				throw new InterestException(InterestExceptionEnum.ERROR_PARAM_MISS);
			}
			Integer interestGoodsId = interestGoodsSkuVO.getInterestGoodsId();
			InterestGoods interestGoods = interestGoodsDao.selectOne(interestGoodsId);
			if(ObjectUtils.isEmpty(interestGoods)) {
				logger.error(InterestExceptionEnum.ERROR_NOT_FOUND.getErrMsg());
				throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND);
			}
			String skuCode = interestGoodsSkuVO.getSkuCode();
			InterestGoodsSku interestGoodsSku = interestGoodsSkuDao.getInterestGoodsSku(interestGoodsId, skuCode);
			if(ObjectUtils.isEmpty(interestGoodsSku)) {
				logger.error(InterestExceptionEnum.ERROR_NOT_FOUND.getErrMsg());
				throw new InterestException(InterestExceptionEnum.ERROR_NOT_FOUND);
			}
			int usedQuantity = interestGoodsProvideDao.countCashQuantity(memberId, interestGoodsId, interestGoodsSku.getId() , false);
			if(interestGoodsSkuVO.getCashQuantity() > usedQuantity) {
				logger.error(InterestExceptionEnum.ERROR_BACK_EXCESS.getErrMsg());
				throw new InterestException(InterestExceptionEnum.ERROR_BACK_EXCESS);
			}
			interestGoodsSkuVO.setInterestGoodsSkuId(interestGoodsSku.getId());
			interestGoodsSkuVO.setProductCode(interestGoods.getProdCode());
		}
		boolean success = false;
		// 修改兑换状态
		success = interestGoodsProvideDao.resetCash(memberId, interestGoodsSkuVOList);
		if(!success) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_STATUS.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_STATUS);
		}
		// 修改兑换数量
		success = interestGoodsSkuDao.reduceCashQuantity(interestGoodsSkuVOList);
		if(!success) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_EXCESS.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_EXCESS);
		}
		// 返还购物车
		success = this.backInterestProductToCart(memberId, interestGoodsSkuVOList);
		if(!success) {
			logger.error(InterestExceptionEnum.ERROR_PUT_CART.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_PUT_CART);
		}
	}
	
	/**
	 * 商品权益加入购物车
	 * @Methods Name backInterestProductToCart
	 * @Create In 2017-8-14 By Xiang.Gao
	 * @param memberId
	 * @param InterestGoodsSkuVOList
	 * @return boolean
	 */
	private boolean backInterestProductToCart(Integer memberId, List<InterestGoodsSkuVO> InterestGoodsSkuVOList) {
		List<CartProductPack> cartProductPackList = Lists.newArrayList();
		
		for (InterestGoodsSkuVO interestGoodsSkuVO : InterestGoodsSkuVOList) {
			CartProductPack cartProductPack = new CartProductPack();
	        cartProductPack.setSource(CartPackSource.INTEREST);
	        cartProductPack.setOwnerId(Long.valueOf(memberId+""));
	        cartProductPack.setFactProductPrice(BigDecimal.ZERO);
	        cartProductPack.setCount(interestGoodsSkuVO.getCashQuantity());
	        //设置spvId
	        PsProductVO psProductVO = mcPsProductService.findProdByCode(interestGoodsSkuVO.getProductCode());
	        for (PsSpvVO spv : psProductVO.getSpvList()) {
	            if (spv.getSkuCode().equalsIgnoreCase(interestGoodsSkuVO.getSkuCode())) {
	                cartProductPack.setSpvId(spv.getSpvId());
	                cartProductPack.setProductId(spv.getProductId());
	                cartProductPack.setSourceBindVal(interestGoodsSkuVO.getInterestGoodsId()+"");
	                break;
	            }
	        }
	        cartProductPackList.add(cartProductPack);
		}
		ServiceResp<Boolean> resp = cartService.batchSaveInterestPack(cartProductPackList);
		return resp.isSuccess();
	}
	
	@Transactional("scmDataSourceTx")
	@Override
	public boolean changeInterestGoodsSku(Integer memberId, Integer sourceInterestGoodsSkuId, Integer destInterestGoodsSkuId) {
		logger.info("change product memberId:{},sourceGoodsSkuId:{}, destGoodsSkuId:{}", 
				memberId,sourceInterestGoodsSkuId,destInterestGoodsSkuId);
		// 减少原商品权益SKU领取数量
		boolean isSuccess = interestGoodsSkuDao.reduceReceiveQuantity(sourceInterestGoodsSkuId);
		if (!isSuccess) {
			logger.error("reduce source goods sku error");
			return false;
		}
		//增加目标商品权益SKU领取数量
		isSuccess = interestGoodsSkuDao.addReceiveQuantity(destInterestGoodsSkuId);
		if (!isSuccess) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_RECEIVE.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_RECEIVE);
		}
		int res = interestGoodsProvideDao.changeInterestGoodsSkuId(memberId, sourceInterestGoodsSkuId, destInterestGoodsSkuId);
		if(res != 1) {
			logger.error(InterestExceptionEnum.ERROR_UPDATE_CHANGE.getErrMsg());
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_CHANGE);
		}
		return true;
	}

	@Override
	public Map<Integer, Integer> getCashQuantity(List<Integer> interestGoodsSkuIds) {
		return interestGoodsProvideDao.getCashQuantity(interestGoodsSkuIds);
	}
	
	@Override
	public List<InterestGoodsSku> getIdByValidity() {
		List<InterestGoods> interestGoodsList = interestGoodsDao.getInterestByTime(DateUtils.getCurrentDate());
		List<InterestGoodsSku> result = new ArrayList<InterestGoodsSku>();
		for (InterestGoods interestGoods : interestGoodsList) {
			result.addAll(interestGoodsSkuDao.byInterestGoodsId(interestGoods.getId()));
		}
		return result;
	}

	@Override
	public List<InterestGoodsGetVO> getPreExpiration(Integer day) {
		List<InterestGoodsGetVO> InterestGoodsGetVOList = Lists.newArrayList();
		List<InterestGoods> interestGoodsList = interestGoodsDao.getInterestByTime(
				DateUtil.addDay(DateUtils.getCurrentDate(), day));
		if(CollectionUtils.isEmpty(interestGoodsList)) {
			return InterestGoodsGetVOList;
		}
		for(InterestGoods interestGoods : interestGoodsList) {
			List<InterestGoodsProvide> interestGoodsProvideList = 
					interestGoodsProvideDao.queryInterestGoodsProvide(interestGoods.getId());
			for(InterestGoodsProvide interestGoodsProvide : interestGoodsProvideList) {
				InterestGoodsSku sku = interestGoodsSkuDao.selectOne(interestGoodsProvide.getInterestGoodsSkuId());
				InterestGoodsGetVO interestGoodsGetVO = new InterestGoodsGetVO(interestGoods, sku);
				interestGoodsGetVO.setMemberId(interestGoodsProvide.getMemberId());
				InterestGoodsGetVOList.add(interestGoodsGetVO);
			}
		}
		return InterestGoodsGetVOList;
	}
	
	@Override
	public void batchUpdateStatus(List<Integer> interestGoodsIds) {
		logger.info("batch update interestGoods status param:", interestGoodsIds);
		
		int res = interestGoodsDao.batchUpdateStatus(interestGoodsIds);
		if (res != interestGoodsIds.size()) {
			logger.error("update interestGoods status error");
			throw new InterestException(InterestExceptionEnum.ERROR_UPDATE_DB);
		}
		logger.info("batch update interestGoods status success");
	}

}
