package com.thebeastshop.pegasus.merchandise.service;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.dubbo.common.utils.StringUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.merchandise.IService.IInterestFreeService;
import com.thebeastshop.pegasus.merchandise.enums.InterestFreeEnum;
import com.thebeastshop.pegasus.merchandise.exception.PublicException;
import com.thebeastshop.pegasus.merchandise.exception.PublicExceptionErrorCode;
import com.thebeastshop.pegasus.merchandise.vo.OpProductVO;
import com.thebeastshop.pegasus.merchandise.vo.OpSpvVO;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuCategoryVO;
import com.thebeastshop.pegasus.merchandise.vo.PcsSkuCombinationVO;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.scm.cond.InterestFreeCond;
import com.thebeastshop.scm.dao.InterestFreeDao;
import com.thebeastshop.scm.dao.InterestFreeProductDao;
import com.thebeastshop.scm.dao.InterestFreeProvideDao;
import com.thebeastshop.scm.dao.MemberDao;
import com.thebeastshop.scm.po.InterestFree;
import com.thebeastshop.scm.po.InterestFreeProduct;
import com.thebeastshop.scm.po.InterestFreeProvide;
import com.thebeastshop.scm.po.Member;
import com.thebeastshop.scm.vo.interest.InterestFreeProductExcelVO;
import com.thebeastshop.scm.vo.interest.InterestFreeProvideVO;
import com.thebeastshop.scm.vo.interest.InterestFreeUseVO;
import com.thebeastshop.scm.vo.interest.InterestFreeVO;

/**
 * 免邮权益Service 实现类
 * @Class Name InterestFreeService
 * @Author Xiang.Gao
 * @Create In 2017-3-16
 */
@Service
public class InterestFreeService implements IInterestFreeService{
	
	private static final Logger logger = LoggerFactory.getLogger(InterestFreeService.class);
	
	private static final Integer ALL_MEMBER_ID = -1;
	
	@Resource
	private InterestFreeDao interestFreeDao;
	
	@Resource
	private InterestFreeProductDao interestFreeProductDao;
	
	@Resource
	private InterestFreeProvideDao interestFreeProvideDao;
	
	@Resource 
	private MemberDao memberDao;
	
	@Resource(name = "mcOpProductService")
	private McOpProductService mcOpProductService;
	
	@Resource(name = "mcPcsSkuCategoryService")
	private McPcsSkuCategoryService mcPcsSkuCategoryService;
	
	@Transactional("scmDataSourceTx")
	@Override
	public void save(InterestFreeVO interestFreeVO) {
		logger.info("[免邮权益] 权益保存  参数:{}",interestFreeVO);
		this.checkInterestFreeNum(interestFreeVO);
		
		// 保存免邮权益主信息
		InterestFree interestFree = InterestFreeVO.toInterstFree(interestFreeVO);
		int rs = interestFreeDao.insertOrUpdate(interestFree);
		if(rs != 1) {
			throw new PublicException(PublicExceptionErrorCode.CREATE_ERROR,"保存权益错误");
		}
		
		// 免邮权益关联操作
		this.interestFreeReletion(interestFreeVO,interestFree.getId());
		logger.info("[免邮权益] 保存成功");
	}

	@Override
	public List<InterestFreeVO> queryList(InterestFreeCond cond) {
		return interestFreeDao.queryList(cond);
	}

	@Override
	public InterestFreeVO queryByInterestFreeId(Integer interestFreeId) {
		InterestFree interestFree = interestFreeDao.selectOne(interestFreeId);
		if(interestFree == null) {
			throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR); 
		}
		
		// 查询适用商品所勾选的类目
		List<Long> categorys = interestFreeProductDao.selectProductCategory(interestFreeId);
		InterestFreeVO interestFreeVO = new InterestFreeVO(interestFree);
		interestFreeVO.setCategories(categorys);
		return interestFreeVO;
	}

	@Override
	public List<InterestFreeProvideVO> selectProvideRecord(InterestFreeCond cond) {
		return interestFreeProvideDao.selectProvideRecord(cond);
	}
	
	@Override
	public List<InterestFreeProductExcelVO> selectProducts(Integer interestFreeId) {
		List<InterestFreeProduct> productList = interestFreeProductDao.selectProduct(interestFreeId);
		OpProductVO opProductVO = null;
		List<InterestFreeProductExcelVO> excelList = Lists.newArrayList();
		for(InterestFreeProduct product : productList) {
			opProductVO = mcOpProductService.findById(product.getBingdingId());
			excelList.add(new InterestFreeProductExcelVO(opProductVO.getCode(),opProductVO.getNameCn()));
		}
		return excelList;
	}
	
	/**-----------------------------------   以下接口提供给外部调用       --------------------------*/

	@Transactional("scmDataSourceTx")
	@Override
	public InterestFreeProvide interestFreeProvide(Integer memberId, Integer interestFreeId, String provideChannel) {
		logger.info("[免邮权益] 权益发放  memberId:{},interestFreeId:{},provideChannel:{}"
				, memberId, interestFreeId, provideChannel);
		
		// 基础信息校验
		if(memberId == null || interestFreeId == null || StringUtils.isBlank(provideChannel)) {
			throw new PublicException("","参数缺失");
		}
		InterestFree interestFree = interestFreeDao.selectOne(interestFreeId);
		if(interestFree == null) {
			throw new PublicException("","未找到ID为["+interestFreeId+"]的免邮权益信息"); 
		}
		Member member = memberDao.selectOne(memberId);
		if(member == null) {
			throw new PublicException("","未找到ID为["+memberId+"]的用户信息"); 
		}
		
		// 权益发放校验
		this.checkProvide(memberId, interestFree, 3);
		
		// 保存权益发放记录
		InterestFreeProvide provide = buildingProvide(interestFreeId, memberId, provideChannel);
		int res = interestFreeProvideDao.insert(provide);
		if(res == 0) {
			throw new PublicException(PublicExceptionErrorCode.CREATE_ERROR,"保存免邮权益发放信息失败"); 
		}
		logger.info("[免邮权益] 权益发放成功 provideId：{}",provide.getId());
		return provide;
	}

	@Override
	public InterestFreeVO getCanUseInterestFree(InterestFreeUseVO interestFreeUseVO) {
		logger.info("[免邮权益] 获取用户可用免邮权益  参数:{}",interestFreeUseVO);
		// 参数校验
		if(interestFreeUseVO == null || interestFreeUseVO.getMemberId() == null 
				|| StringUtils.isBlank(interestFreeUseVO.getAccessWay())
				|| CollectionUtils.isEmpty(interestFreeUseVO.getProductCodes())) {
			throw new PublicException("","缺少必要参数");
		}
		String accessWay = InterestFreeEnum.getCode(interestFreeUseVO.getAccessWay().toUpperCase())+"";
		if(StringUtils.isBlank(accessWay)) {
			logger.info("[免邮权益] 未能解析到适用平台");
			return null;
		}
		
		Member member = memberDao.selectOne(interestFreeUseVO.getMemberId());
		if(member == null) {
			logger.info("[免邮权益] memberId为："+ interestFreeUseVO.getMemberId() +"用户不存在!");
			return null;
		}
		
		List<String> productCodes = interestFreeUseVO.getProductCodes();
		// 是否存在海淘商品
		boolean existsCrossBorder = this.existsCrossBorder(productCodes);
		int excluBorderFlag = existsCrossBorder ? InterestFreeEnum.CROSS_BORDER_INLAND.getCode()
				: InterestFreeEnum.CROSS_BORDER_OVERSEAS.getCode() ;
		// 获取用户所有权益信息
		Integer memberId = interestFreeUseVO.getMemberId();
		List<InterestFreeVO> allInterestFree = interestFreeDao.selectByMemberId(
				memberId, accessWay, excluBorderFlag);
		// 海淘免邮福利查询
		queryCrossBorderFree(member.getMemberLevel(), existsCrossBorder, allInterestFree);
		
		if(CollectionUtils.isEmpty(allInterestFree)) {
			logger.info("[免邮权益] 会员["+memberId+"]未找到可用的免邮权益");
			return null;
		}
		// 匹配可用免邮权益
		List<InterestFreeVO> canUseInterestFree = matchInterestFree(productCodes, allInterestFree);
		if(CollectionUtils.isEmpty(canUseInterestFree)) {
			logger.info("[免邮权益] 未能匹配到可用权益");
			return null;
		}
		InterestFreeVO interestFreeVO = sortInterestFree(canUseInterestFree);
		logger.info("[免邮权益] 获取用户可用免邮权益 :{}",interestFreeVO);
		return interestFreeVO;
	}
	
	@Override
	public InterestFreeVO queryInterestFreeInfo(Integer provideId) {
		logger.info("[免邮权益] 权益详情  provideId:{}",provideId);
		if(provideId == null) {
			throw new PublicException("","缺少必要参数");
		}
		
		InterestFreeVO vo = interestFreeDao.selectByProvideId(provideId);
		if(vo == null) {
			throw new PublicException("","未找到provideID为["+provideId+"]的免邮权益信息"); 
		}
		vo.setProductScopeName(InterestFreeEnum.getDesc("productScope",vo.getProductScope()));
		if(InterestFreeEnum.USE_TIME_FIXED.getCode().equals(vo.getUseTimeType())) {
			vo.setValidateDateStr(DateUtil.format(vo.getUseBeginDate(), DateUtil.DEFAULT_DATETIME_FORMAT) +
					"至" + DateUtil.format(vo.getUseEndDate(), DateUtil.DEFAULT_DATETIME_FORMAT));
		} else {
			vo.setValidateDateStr("领取即日起"+vo.getProvideAfterDay()+"天以内");
			vo.setUseBeginDate(vo.getProvideTime());
			vo.setUseEndDate(DateUtil.addDay(vo.getProvideTime(), vo.getProvideAfterDay()));
		}
		
		vo.setFreeStatus(InterestFreeEnum.FREE_STATUS_VALID.getCode());
		// 已过期
		if(vo.getUseEndDate().before(DateUtil.getNow())) {
			vo.setFreeStatus(InterestFreeEnum.FREE_STATUS_EXPIRED.getCode());
		}
		if(vo.getFreePostNum() != 0 && vo.getUseTotal() >= vo.getFreePostNum()) {
			vo.setFreeStatus(InterestFreeEnum.FREE_STATUS_USEUP.getCode());
		}
		logger.info("[免邮权益] 权益详情 end InterestFreeVO:{}",vo);
		return vo;
	}
	
	@Override
	public void interestFreeCash(InterestFreeUseVO vo) {
		logger.info("[免邮权益] 权益兑换 参数:{}",vo);
		// 基础信息校验
		if(vo == null || vo.getProvideId() == null || vo.getMemberId() == null 
				|| StringUtils.isBlank(vo.getAccessWay())){
			throw new PublicException("","缺少必要参数");
		}
		InterestFreeVO interestFree = checkInterestFreeUse(vo);
		// 开始使用免邮权益
		this.usedInterestFree(interestFree, 3);
		logger.info("[免邮权益] 权益兑换成功");
	}
	
	@Override
	public void interestFreeBack(InterestFreeUseVO useVO) {
		logger.info("[免邮权益] 权益返还 参数:{}",useVO.toString());
		if(useVO == null || useVO.getProvideId() == null) {
			throw new PublicException("","缺少必要参数");
		}
		InterestFreeProvide provide = interestFreeProvideDao.selectOne(useVO.getProvideId());
		if(provide == null) {
			throw new PublicException("","未能找到免邮权益使用记录");
		}
		int res = interestFreeProvideDao.backInterestFree(useVO.getProvideId());
		if(res != 1) {
			throw new PublicException(PublicExceptionErrorCode.UPDATE_ERROR,"返还免邮权益使用数量失败");
		}
		logger.info("[免邮权益] 权益返回成功");
	}
	
	@Override
	public List<InterestFreeVO> getPreExpiration(Integer day) {
		logger.info("[免邮权益] 查询["+day+"]天后到期的免邮权益");
		if(day == null) {
			throw new PublicException("","缺少必要参数");
		}
		List<InterestFreeVO> interestFreeVOList =  interestFreeDao.selectAllMemberInterestFree();
		if(CollectionUtils.isEmpty(interestFreeVOList)) {
			logger.info("[免邮权益] 未能查询到已发放未使用的免邮权益");
			return interestFreeVOList;
		}
		List<InterestFreeVO> remindFreeList = Lists.newArrayList();
		for(InterestFreeVO vo : interestFreeVOList) {
			if (ALL_MEMBER_ID.equals(vo.getMemberId())) {
				continue;
			}
			if(InterestFreeEnum.USE_TIME_AFTER.getCode().equals(vo.getUseTimeType())) {
				vo.setUseEndDate(DateUtil.addDay(vo.getProvideTime(), vo.getProvideAfterDay()));
			} 
			Date remindDate = DateUtil.addDay(DateUtil.getNow(), day);
			if(remindDate.after(vo.getUseEndDate())) {
				remindFreeList.add(vo);
				logger.info("[免邮权益] 用户:[" + vo.getMemberId() + "]发放的["+vo.getProvideId()+"]的免邮权益将于[" 
				+ DateUtil.format(vo.getUseEndDate(), DateUtil.DEFAULT_DATETIME_FORMAT) + "]到期");
			}
		}
		return remindFreeList;
	}
	
	/**
	 * 获取海淘免邮权益(针对老虎，大象会员 所有用户 ) 
	 * @Methods Name queryCrossBorderFree
	 * @Create In 2017-4-23 By Administrator
	 */
	private void queryCrossBorderFree(Integer memberLevel, 
			boolean existsCrossBorder, List<InterestFreeVO> allInterestFree) {
		if(existsCrossBorder && 
				(Integer.valueOf(2).equals(memberLevel) || Integer.valueOf(3).equals(memberLevel))) {
			List<InterestFreeVO> crossBorderFree = interestFreeDao.selectCrossBorderInterestFree(ALL_MEMBER_ID);
			allInterestFree.addAll(crossBorderFree);
		}
	}
	
	/**
	 * 匹配用户可用免邮权益
	 * @Methods Name matchInterestFree
	 * @Create In 2017-3-23 By Xiang.Gao
	 */
	private List<InterestFreeVO> matchInterestFree(List<String> productCodes, 
			List<InterestFreeVO> allInterestFree) {
		logger.info("[免邮权益] 开始匹配可用免邮权益 "+JSON.toJSONString(allInterestFree));
		// 可用权益
		List<InterestFreeVO> canUseInterestFree = Lists.newArrayList();
		// 全场商品免邮权益
		List<InterestFreeVO> allScopeInterestFree = Lists.newArrayList();
		// 指定商品免邮权益
		List<InterestFreeVO> fixedScopeInterestFree = Lists.newArrayList();
		
		this.interestFreeWash(fixedScopeInterestFree, allScopeInterestFree, allInterestFree);
		// 免邮权益  全场商品查询排除关系
		if(CollectionUtils.isNotEmpty(allScopeInterestFree)) {
			List<Integer> bingdingIds = getBingdingIds(productCodes, false);
			for(InterestFreeVO vo : allScopeInterestFree) {
				boolean exclude = interestFreeDao.selectRelation(vo.getId(), bingdingIds,
						InterestFreeEnum.IS_EXCLUDE_TRUE.getCode());
				if(exclude) {
					continue;
				}
				canUseInterestFree.add(vo);
			}
		}
		// 免邮权益 指定商品 查询适用关系
		if(CollectionUtils.isNotEmpty(fixedScopeInterestFree)) {
			List<Integer> bingdingIds = getBingdingIds(productCodes, true);
			for(InterestFreeVO vo : fixedScopeInterestFree) {
				boolean isExsts = interestFreeDao.selectRelation(vo.getId(), bingdingIds,
						InterestFreeEnum.IS_EXCLUDE_FALSE.getCode());
				if(!isExsts){
					continue;
				}
				canUseInterestFree.add(vo);
			}
		}
		logger.info("[免邮权益] 匹配可用免邮权益结果 :"+JSON.toJSONString(canUseInterestFree));
		return canUseInterestFree;
	}
	
	/**
	 * 是否存在海淘商品  并返回所排除的限制标示
	 * @Methods Name existsCrossBorder
	 * @Create In 2017-3-24 By Xiang.Gao
	 */
	private boolean existsCrossBorder(List<String> productCodes) {
		boolean isCrossBorder  = false;
		for(String productCode : productCodes) {
			isCrossBorder = mcOpProductService.getCrossBorderFlagListByCode(productCode, 0);
			if(isCrossBorder){
				isCrossBorder = true;
				break;
			}
		}
		return isCrossBorder ;
	}
	
	/**
	 * 免邮权益筛选
	 * @Methods Name interestFreeWash
	 * @Create In 2017-3-23 By Xiang.Gao
	 */
	private void interestFreeWash(List<InterestFreeVO> fixedScopeInterestFree, 
			List<InterestFreeVO> allScopeInterestFree,List<InterestFreeVO> allInterestFree) {
		for(InterestFreeVO interestFree : allInterestFree) {
			if(InterestFreeEnum.USE_TIME_FIXED.getCode().equals(interestFree.getUseTimeType())) {
				// 固定时间
				Date toDay = DateUtil.getNow();
				if(interestFree.getUseBeginDate().after(toDay) 
						|| interestFree.getUseEndDate().before(toDay)) {
					continue;
				}
			} else {
				// 领取后N天
				Date useEndDay = DateUtil.addDay(interestFree.getProvideTime(), interestFree.getProvideAfterDay());
				if(useEndDay.before(DateUtil.getNow())) {
					continue;
				}
			}
			// 全场商品/指定商品处理 productScope 0 ：指定,1：全场
			if(InterestFreeEnum.PRODUCT_SCOPE_ALL.getCode().equals(interestFree.getProductScope())) {
				allScopeInterestFree.add(interestFree);
			} else {
				fixedScopeInterestFree.add(interestFree);
			}
		}
	}
	
	/**
	 * 可用免邮权益排序并返回第一个
	 * 
	 * 1）到期日近>到期日远
	 * 2）指定商场>全场
	 * 3）仅限海淘/非海淘>不限制
	 * 4）使用次数由少到多
	 * 5）发放到用户账号时间由早到晚
	 * 
	 * @Methods Name sortInterestFree
	 * @Create In 2017-3-23 By Xiang.Gao
	 */
	private InterestFreeVO sortInterestFree(List<InterestFreeVO> list) {
		logger.info("[免邮权益] 可用免邮权益排序:"+JSON.toJSONString(list));
		InterestFreeVO vo = list.get(0);
		for(InterestFreeVO interestFree : list) {
			// 到期日近>到期日远
			Date firstDate = InterestFreeEnum.USE_TIME_FIXED.getCode().equals(vo.getUseTimeType()) 
					? vo.getUseEndDate() 
					: DateUtil.addDay(vo.getProvideTime(),vo.getProvideAfterDay());
			Date secondDate = InterestFreeEnum.USE_TIME_FIXED.getCode().equals(interestFree.getUseTimeType())
					? interestFree.getUseEndDate()
					: DateUtil.addDay(interestFree.getProvideTime(), interestFree.getProvideAfterDay());
			if(firstDate.after(secondDate)) {
				vo = interestFree;
				continue;
			}
			// 指定商场>全场
			if(InterestFreeEnum.PRODUCT_SCOPE_ALL.getCode().equals(vo.getProductScope()) 
					&& InterestFreeEnum.PRODUCT_SCOPE_FIXED.getCode().equals(interestFree.getProductScope())) {
				vo = interestFree;
				continue;
			}
			// 仅限海淘/非海淘>不限制
			if(InterestFreeEnum.CROSS_BORDER_NOLIMIT.getCode().equals(vo.getCrossBorderFlag()) 
					&& !InterestFreeEnum.CROSS_BORDER_NOLIMIT.getCode().equals(interestFree.getCrossBorderFlag())) {
				vo = interestFree;
				continue;
			}
			// 使用次数由少到多
			if(interestFree.getUseTotal() < vo.getUseTotal()) {
				vo = interestFree;
				continue;
			}
			// 发放到用户账号时间由早到晚
			if(interestFree.getProvideTime().before(vo.getProvideTime())) {
				vo = interestFree;
				continue;
			}
		}
		return vo;
	}
	
	/**
	 * 权益相关数量变更校验
	 * @Methods Name checkInterestFreeNum
	 * @Create In 2017-3-23 By Xiang.Gao
	 * @param vo void
	 */
	private void checkInterestFreeNum(InterestFreeVO vo) {
		logger.info("[免邮权益] 保存 权益相关数量验证 ");
		Integer interestFreeId = vo.getId();
		if(vo.getFreePostNum() != null) {
			int maxUsedNum = interestFreeProvideDao.selectMaxUsedNum(interestFreeId);
			if(vo.getFreePostNum() < maxUsedNum) {
				throw new PublicException("","免邮次数不小已使用最大次数："+maxUsedNum);
			}
		}
		if(vo.getReceiveNum() != null) { 
			int receiveTotal = interestFreeProvideDao.countReceive(interestFreeId);
			if(vo.getReceiveNum() < receiveTotal) {
				throw new PublicException("","限领数量不得小于已领取数量:"+receiveTotal);
			}
		}
		if(vo.getSignReceiveNum() != null) {
			int signReceiveNum = interestFreeProvideDao.selectSignMaxReceiveNum(interestFreeId);
			if(vo.getSignReceiveNum() < signReceiveNum){
				throw new PublicException("","单个用户限领数不得小于目前已领取数"+signReceiveNum);
			}
		}
		logger.info("[免邮权益] 保存 权益相关数量验证通过 ");
	}
	
	/**
	 * 权益关联操作
	 * @Methods Name interestFreeReletion
	 * @Create In 2017-3-21 By Xiang.Gao
	 */
	private void interestFreeReletion(InterestFreeVO vo, Integer interestFreeId) {
		if(vo.getId() != null) {
			// 修改权益  先删除原适用商品 再保存新的适用商品
			Integer bingdingType = null;
			if(CollectionUtils.isNotEmpty(vo.getCategories())) {
				bingdingType = InterestFreeEnum.BINGDING_TYPE_CATEGORY.getCode();
				interestFreeProductDao.batchDelete(vo.getId(), bingdingType);
			}
			if(CollectionUtils.isNotEmpty(vo.getInterestFreeProduct())) {
				bingdingType = InterestFreeEnum.BINGDING_TYPE_PRODUCT.getCode();
				interestFreeProductDao.batchDelete(vo.getId(), bingdingType);
			}
		}
		vo.setId(interestFreeId);
		if(CollectionUtils.isNotEmpty(vo.getInterestFreeProduct())) {
			this.buildingProduct(vo);
		}
		if(CollectionUtils.isNotEmpty(vo.getCategories())) {
			this.buildingCategory(vo);
		}
		if(CollectionUtils.isNotEmpty(vo.getInterestFreeProduct())) {
			// 批量保存
			int rs = interestFreeProductDao.batchInsert(vo.getInterestFreeProduct());
			if(rs != vo.getInterestFreeProduct().size()) {
				throw new PublicException(PublicExceptionErrorCode.CREATE_ERROR, "保存适用商品出错");
			}
		}
	}
	
	/**
	 * 获取商品Id及商品下SKU所对应类目ID
	 * @Methods Name getBingdingIds
	 * @Create In 2017-3-20 By Xiang.Gao
	 */
	private List<Integer> getBingdingIds(List<String> productCodes, boolean needCategory) {
		logger.info("[免邮权益] 获取适用商品绑定ID productCodes:{}",JSON.toJSONString(productCodes));
		List<Integer> bingdingIds = Lists.newArrayList();
		for(String productCode : productCodes) {
			// 查询商品信息
			OpProductVO productVO = mcOpProductService.findProductByCode(productCode);
			if(productVO == null || productVO.getId() == null) {
				throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR,
						"未找到CODE：["+productCode+"]的商品信息");
			}
			
			// 绑定商品ID
			bingdingIds.add(Integer.parseInt(productVO.getId()+""));
			if(!needCategory) {
				continue;
			}
			// 获取Product下所有SkuCode
			List<String> skuCodes = getProductSkuCodes(productVO, productCode);
			// 获取Sku所对应的类目ID
			for(String skuCode : skuCodes ) {
				List<PcsSkuCategoryVO> skuCategoryList = mcPcsSkuCategoryService
						.findBackCategoriesBySkuCode(skuCode);
				if(CollectionUtils.isEmpty(skuCategoryList)){
					throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR,
							"未找到CODE：["+productCode+"]的商品类目信息");
				}
				for(PcsSkuCategoryVO skuCategory : skuCategoryList) {
					bingdingIds.add(Integer.parseInt(skuCategory.getId()+""));
				}
			}
		}
		logger.info("[免邮权益] 获取适用商品绑定ID:{}",JSON.toJSONString(bingdingIds));
		return bingdingIds;
	}
	
	/**
	 * 获取Product下所有SkuCode
	 * @Methods Name getProductSkuCodes
	 * @Create In 2017-3-23 By Xiang.Gao
	 */
	private List<String> getProductSkuCodes(OpProductVO productVO,String productCode) {
		// 绑定商品下SKU所对应的类目ID
		List<String> skuCodes = Lists.newArrayList();
		if(productVO.getCombined()){
			// 组合商品 默认取 Product下第一个Spv 下第一个Sku
			OpProductVO product = mcOpProductService.findCombinedProdbyProdId(productVO.getId());
			if(product != null && CollectionUtils.isNotEmpty(product.getSpvList())){
				OpSpvVO spvVO = product.getSpvList().get(0);
				List<PcsSkuCombinationVO> list = spvVO.getSkuCombinationList();
				if(CollectionUtils.isNotEmpty(list)) {
					PcsSkuCombinationVO combinationVO = list.get(0);
					skuCodes.add(combinationVO.getSubSkuCode());
				}
			}
		}else {
			// 普通商品
			List<OpProductVO> productList = mcOpProductService.findProdWithSpvByNameOrCode(productCode);
			if(CollectionUtils.isNotEmpty(productList)) {
				for(OpProductVO opProductVO : productList) {
					skuCodes.addAll(opProductVO.getSkuCodes());
				}
			}
		}
		logger.info("[免邮权益] 获取product下所对应的SKUCode:{}",JSON.toJSONString(skuCodes));
		return skuCodes;
	}
	
	/**
	 * 权益发放校验
	 * @Methods Name checkProvide
	 * @Create In 2017-3-20 By Xiang.Gao
	 */
	private void checkProvide(Integer memberId, InterestFree interestFree,Integer retryTotal) {
		logger.info("[免邮权益] 权益发放开始校验");
	    // 单个用户领取数量检验
		Integer interestFreeId = interestFree.getId();
 		int signReceiveNum = interestFreeProvideDao.countReceiveByMember(interestFreeId, memberId);
 		if(interestFree.getSignReceiveNum() != 0 
 				&& interestFree.getSignReceiveNum() <= signReceiveNum) {
 			throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR,
 					"ID为["+interestFreeId+"]的免邮权益每个用户限领["+interestFree.getSignReceiveNum()+"]次"); 
 		}
 		// 领取数量校验
 		int receivedNum = interestFreeProvideDao.countReceive(interestFreeId);
 		if(interestFree.getReceiveNum() != 0 && interestFree.getReceiveNum() <= receivedNum) {
 			throw new PublicException(PublicExceptionErrorCode.SEARCH_ERROR,
 					"ID为["+interestFreeId+"]的免邮权益已被领取完"); 
 		}
 		// 更新免邮权益领取总数 获取 数据库乐观锁 
	    int res = interestFreeDao.updateReceived(interestFree.getId(), interestFree.getReceivedTotal());
	    if(res == 0 ){
	    	logger.info("[免邮权益] 权益发放校验 数据锁失败,将重试{}"+retryTotal);
	    	// 锁失败 递归重试3次
	    	if(retryTotal != 0) {
		    	retryTotal --;
		    	this.checkProvide(memberId, interestFree, retryTotal);
	    	} else {
	    		logger.error("[免邮权益] 权益发放校验 获取数据库锁重试3次失败");
	    		throw new PublicException("","免邮权益领取失败");
	    	}
	    }
	    logger.info("[免邮权益] 权益发放校验通过");
	}
	
	/**
	 * 权益使用校验
	 * @Methods Name checkInterestFreeUse
	 * @Create In 2017-3-20 By Xiang.Gao
	 */
	private InterestFreeVO checkInterestFreeUse(InterestFreeUseVO vo) {
		logger.info("[免邮权益] 权益兑换开始校验");
		InterestFreeVO interestFree = interestFreeDao.selectByProvideId(vo.getProvideId());
		if(interestFree == null) {
			throw new PublicException("","未找到provideId为["+vo.getProvideId()+"]的免邮权益发放记录"); 
		}
		// 适用平台检验
		String accessWay = InterestFreeEnum.getCode(vo.getAccessWay().toUpperCase())+"";
		if(!interestFree.getAccessWay().contains(accessWay)) {
			throw new PublicException("","ID为["+interestFree.getId()+"]的免邮权益适用平台不符合");
		}
		// 使用时间检验
		if(InterestFreeEnum.USE_TIME_FIXED.getCode().equals(interestFree.getUseTimeType())) {
			// 固定时间
			Date currDay = DateUtil.getNow();
			if(interestFree.getUseBeginDate().after(currDay)
					|| interestFree.getUseEndDate().before(currDay)) {
				throw new PublicException("","ID为["+interestFree.getId()+"]的免邮权益不在使用时间范围内");
			}
		} else {
			// 领取后N天
			Date useEndTime = DateUtil.addDay(interestFree.getProvideTime(), interestFree.getProvideAfterDay());
			if(DateUtil.getNow().after(useEndTime)) {
				throw new PublicException("","ID为["+interestFree.getId()+"]的免邮权益已过期");
			}
		}
		logger.info("[免邮权益] 权益兑换校验通过");
		return interestFree;
	}
	
	/**
	 * 使用免邮权益
	 * @Methods Name usedInterestFree
	 * @Create In 2017-3-27 By Xiang.Gao
	 */
	private void usedInterestFree(InterestFreeVO vo, Integer retry) {
		// 使用限制次数校验
		InterestFreeProvide provide = interestFreeProvideDao.selectOne(vo.getProvideId());
		if(vo.getFreePostNum() != 0 && provide.getUsedNum() >= vo.getFreePostNum()) {
			throw new PublicException("","免邮权益使用超限");
		}
		int res = interestFreeProvideDao.updateUsedNum(vo.getProvideId(), provide.getUsedNum());
		if(res == 0) {
			if(retry != 0) {
				retry --;
				this.usedInterestFree(vo, retry);
			}else {
				logger.error("[免邮权益] 权益使用失败 获取数据库锁重试3次失败");
	    		throw new PublicException("","免邮权益兑换失败");
			}
		}
	}
	
	/**
     * 构建商品集合
     * @Methods Name buildingProduct
     * @Create In 2017-3-20 By Xiang.Gao
     */
    private void buildingProduct(InterestFreeVO vo) {
    	logger.info("[免邮权益] 保存 构建商品集合");
    	List<InterestFreeProduct> productList = Lists.newArrayList();
    	List<InterestFreeProduct> productIds = vo.getInterestFreeProduct();
    	for (InterestFreeProduct product : productIds) {
    		InterestFreeProduct interestFreeProduct = new InterestFreeProduct();
    		interestFreeProduct.setInterestFreeId(vo.getId());
    		// 适用商品类型  0:指定商品 ,1:全场商品  当全场商品时 此处为排除商品  当指定商品时此处为适用商品
    		interestFreeProduct.setIsExclusion(vo.getProductScope());
			// 绑定类型为商品(0)
			Integer bingdingType = InterestFreeEnum.BINGDING_TYPE_PRODUCT.getCode();
    		interestFreeProduct.setBingdingType(bingdingType);
    		interestFreeProduct.setBingdingId(product.getBingdingId());
    		interestFreeProduct.setCreateUser(vo.getCreateUser());
    		interestFreeProduct.setCreateTime(DateUtil.getNow());
    		productList.add(interestFreeProduct);
    	}
    	vo.setInterestFreeProduct(productList);
    	logger.info("[免邮权益] 保存 构建商品集合:"+vo.toString());
    }
	
	 /**
     * 构建商品类目
     * @Methods Name buildingCategory
     * @Create In 2017-3-20 By Xiang.Gao void
     */
    private void buildingCategory(InterestFreeVO vo) {
    	logger.info("[免邮权益] 保存 构建类目集合");
    	List<Long> categories = vo.getCategories();
		InterestFreeProduct interestFreeProduct = null;
		List<InterestFreeProduct> productList = Lists.newArrayList();
		for(Long categorie : categories) {
    		interestFreeProduct = new InterestFreeProduct();
    		interestFreeProduct.setInterestFreeId(vo.getId());
    		// 适用商品类型  0:指定商品 ,1:全场商品  当全场商品时 此处为排除商品  当指定商品时此处为适用商品
			interestFreeProduct.setIsExclusion(vo.getProductScope());
			// 绑定类型为类目(1)
			Integer bingdingType = InterestFreeEnum.BINGDING_TYPE_CATEGORY.getCode();
    		interestFreeProduct.setBingdingType(bingdingType);
    		interestFreeProduct.setBingdingId(categorie);
    		interestFreeProduct.setCreateUser(vo.getCreateUser());
    		interestFreeProduct.setCreateTime(DateUtil.getNow());
    		productList.add(interestFreeProduct);
		}
		if(CollectionUtils.isNotEmpty(vo.getInterestFreeProduct())) {
			vo.getInterestFreeProduct().addAll(productList);
		} else {
			vo.setInterestFreeProduct(productList);
		}
		logger.info("[免邮权益] 保存 构建类目集合:"+vo.toString());
    }
    
	/**
	 * 构建免邮权益方法记录
	 * @Methods Name buildingProvide
	 * @Create In 2017-3-23 By Xiang.Gao
	 */
	private InterestFreeProvide buildingProvide(Integer interestFreeId,Integer memberId,String channel) {
		InterestFreeProvide provide = new InterestFreeProvide();
		provide.setInterestFreeId(interestFreeId);
		provide.setMemberId(memberId);
		provide.setUsedNum(0);
		provide.setProvideChannel(channel);
		provide.setCreateTime(DateUtil.getNow());
		return provide;
	}
	
}
