package com.thebeastshop.pegasus.merchandise.domain.impl;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.merchandise.client.CampaignESClient;
import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.constants.SearchType;
import com.thebeastshop.pegasus.merchandise.domain.CampaignDomain;
import com.thebeastshop.pegasus.merchandise.vo.PsAdvanceCampaignVO;
import com.thebeastshop.pegasus.merchandise.vo.PsCampaignVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import org.apache.commons.collections.CollectionUtils;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @author gongjun
 * @since 2017-01-04
 */
@Component("campaignDomain")
public class CampaignDomainImpl implements CampaignDomain {

    @Autowired
    private CampaignESClient campaignClient;


    @Override
    public List<PsCampaignVO> getGlobalCampaignList(PsMatchCampaignCond cond) {
        cond.setProductScopeIds(Lists.newArrayList(3));
        cond.setStates(Lists.newArrayList(1));
        return getMatchedCampaignList(cond);
    }


    public List<PsCampaignVO> getMatchedCampaignList(PsMatchCampaignCond cond) {
        Integer memberLevel = cond.getMemberLevel();
        Integer accessWay = cond.getAccessWay();

        Map<String, Object> params = new HashMap<>();
        params.put("pagenum", 999);
        if (CollectionUtils.isNotEmpty(cond.getCampaignIds())) {
            params.put("id", cond.getCampaignIds());
        }
        if (CollectionUtils.isNotEmpty(cond.getStates())) {
            params.put("state", QueryBuilders.termsQuery("state", cond.getStates()));
        }
        if (CollectionUtils.isNotEmpty(cond.getDiscountTypeIds())) {
            params.put("campaignList.discountTypeId",
                    QueryBuilders.termsQuery("campaignList.discountTypeId", cond.getDiscountTypeIds()));
        }
        if (CollectionUtils.isNotEmpty(cond.getProductScopeIds())) {
            params.put("productScopeId", QueryBuilders.termsQuery("productScopeId", cond.getProductScopeIds()));
        }
        params.put("accessWayIds", QueryBuilders.termsQuery("accessWayIds", Lists.newArrayList(accessWay)));
        params.put("memberLevels", QueryBuilders.termsQuery("memberLevels", Lists.newArrayList(memberLevel)));
        Date now = new Date();
        params.put("startTime", QueryBuilders.rangeQuery("startTime").lte(now));
        params.put("expireTime", QueryBuilders.rangeQuery("expireTime").gt(now));
        params.put("active", QueryBuilders.termQuery("active", 1));
        List<PsCampaignVO> campaigns = campaignClient.findByFields(params, SearchType.ANDLIKE);
        return campaigns;
    }


    /**
     * 真正匹配活动方法
     * @param campaignList 所有活动集合
     * @param accessWayId   匹配的适用平台
     * @param memberLevelId 匹配的会员等级
     * @param crossBorderFlag  匹配的海涛类型
     * @param matchedAdvanceCampaignList 已匹配的预告活动集合
     * @param matchedCampaignList 已匹配的活动集合
     */
    private void matchCampaignList(List<PsCampaignVO> campaignList, Integer accessWayId, Integer memberLevelId,
                                   Boolean crossBorderFlag, List<PsCampaignVO> matchedAdvanceCampaignList,
                                   List<PsCampaignVO> matchedCampaignList) {
        for (PsCampaignVO campaignVO : campaignList) {
            if (campaignVO.match(accessWayId, memberLevelId, crossBorderFlag)) {
                matchedCampaignList.add(campaignVO);
            } else if (campaignVO.isMatchAdvance(accessWayId, memberLevelId, crossBorderFlag)) {
                matchedAdvanceCampaignList.add(campaignVO);
            }
        }
    }


    /**
     * 匹配商品列表页活动
     * @param productVOList
     * @param cond
     */
    public void matchListPageCampaigns(List<PsProductVO> productVOList, PsMatchCampaignCond cond) {
        if(CollectionUtils.isNotEmpty(productVOList)) {
            List<PsCampaignVO> globalCampaignList = getGlobalCampaignList(cond);
            for (PsProductVO psProductVO : productVOList) {
                matchCampaigns(psProductVO, cond, globalCampaignList);
            }
        }
    }

    /**
     * 对商品中的campaignList以及商品所关联的spv中的campaignList进行匹配
     * @param psProductVO
     * @param cond
     */
    @Override
    public void matchCampaigns(PsProductVO psProductVO, PsMatchCampaignCond cond) {
        List<PsCampaignVO> globalCampaignList = getGlobalCampaignList(cond);
        matchCampaigns(psProductVO, cond, globalCampaignList);
    }

    private void matchCampaigns(PsProductVO psProductVO, PsMatchCampaignCond cond, List<PsCampaignVO> globalCampaigns) {
        List<PsCampaignVO> campaignList = psProductVO.getCampaignList();
        Boolean matchGlobalCampaigns = cond.getMatchGlobalCampaigns();
        // 如果匹配全场，则把全场Product维度活动添加到活动列表
        if (CollectionUtils.isNotEmpty(globalCampaigns)) {
            campaignList.addAll(globalCampaigns);
        }
        Integer memberLevel = cond.getMemberLevel();
        Integer accessWay = cond.getAccessWay();
        Integer accessWayId = (accessWay == null ? -1 : accessWay.intValue());
        Integer memberLevelId = (memberLevel == null ? -1 : memberLevel.intValue());
        Integer crossBorderFlagValue = psProductVO.getCrossBorderFlag();
        Boolean crossBorderFlag = (crossBorderFlagValue != null && crossBorderFlagValue.equals(1));
        if(CollectionUtils.isNotEmpty(campaignList)) {

            List<PsCampaignVO> matchedAdvanceCampaignList = Lists.newArrayList();
            List<PsCampaignVO> matchedProductCampaignList = Lists.newArrayList();

            // 匹配进行中和预告的Product维度的活动
            matchCampaignList(campaignList, accessWayId, memberLevelId, crossBorderFlag,
                    matchedAdvanceCampaignList, matchedProductCampaignList);
            // 设置匹配的进行中Product维度活动
            psProductVO.setCampaignList(matchedProductCampaignList);

            // 设置匹配的预告活动
            if(CollectionUtils.isNotEmpty(matchedAdvanceCampaignList)) {
                PsAdvanceCampaignVO psAdvanceCampaignVO = new PsAdvanceCampaignVO();
                psAdvanceCampaignVO.setAdvanceCampaignList(matchedAdvanceCampaignList);
                psProductVO.setPsAdvanceCampaignVO(psAdvanceCampaignVO);
            }
        }

        List<PsSpvVO> spvList = psProductVO.getSpvList();
        if(CollectionUtils.isNotEmpty(spvList)) {
            for (PsSpvVO spvVO : spvList) {
                matchCampaigns(spvVO, cond, crossBorderFlagValue);
            }
        }
    }

    /**
     * 对spv中的campaignlist进行匹配
     * @param spvVO
     * @param cond
     * @param crossBorderFlagValue
     */
    public void matchCampaigns(PsSpvVO spvVO, PsMatchCampaignCond cond, Integer crossBorderFlagValue) {
        Integer memberLevel = cond.getMemberLevel();
        Integer accessWay = cond.getAccessWay();
        Integer accessWayId = (accessWay == null ? -1 : accessWay.intValue());
        Integer memberLevelId = (memberLevel == null ? -1 : memberLevel.intValue());
        Boolean crossBorderFlag = (crossBorderFlagValue != null && crossBorderFlagValue.equals(1));
        List<PsCampaignVO> campaignList = spvVO.getCampaignList();
        if(CollectionUtils.isNotEmpty(campaignList)) {
            List<PsCampaignVO> matchedAdvanceSpvCampaignList = Lists.newArrayList();
            List<PsCampaignVO> matchedSpvCampaignList = Lists.newArrayList();
            // 匹配进行中和预告的SKU维度活动
            matchCampaignList(campaignList, accessWayId, memberLevelId, crossBorderFlag,
                    matchedAdvanceSpvCampaignList, matchedSpvCampaignList);
            // 设置匹配的进行中活动
            spvVO.setCampaignList(matchedSpvCampaignList);
            // 设置匹配的预告活动
            spvVO.setAdvanceCampaignList(matchedAdvanceSpvCampaignList);
        }
    }
}
