/*
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.merchandise.domain.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.merchandise.cond.OpAttributeCond;
import com.thebeastshop.pegasus.merchandise.cond.OpProductCond;
import com.thebeastshop.pegasus.merchandise.cond.OpSpvCond;
import com.thebeastshop.pegasus.merchandise.conf.PropertyConfigurer;
import com.thebeastshop.pegasus.merchandise.dao.OpProductMapper;
import com.thebeastshop.pegasus.merchandise.domain.*;
import com.thebeastshop.pegasus.merchandise.exception.OperationException;
import com.thebeastshop.pegasus.merchandise.exception.OperationExceptionErrorCode;
import com.thebeastshop.pegasus.merchandise.model.OpChannel;
import com.thebeastshop.pegasus.merchandise.model.OpChnCanSaleProd;
import com.thebeastshop.pegasus.merchandise.model.OpProdSku;
import com.thebeastshop.pegasus.merchandise.model.OpProduct;
import com.thebeastshop.pegasus.merchandise.model.OpProductExample;
import com.thebeastshop.pegasus.merchandise.model.OpSpvPriceChange;
import com.thebeastshop.pegasus.merchandise.vo.*;
import com.thebeastshop.pegasus.util.PegasusConstants;
import com.thebeastshop.pegasus.util.comm.BeanUtil;
import com.thebeastshop.pegasus.util.comm.CodeGenerator;
import com.thebeastshop.pegasus.util.comm.SQLUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @Author: LK【kai.li@thebeastshop.com】
 * @Date: 2016/9/5 13:27
 */
@Component("opProductDomain")
public class OpProductDomainImpl extends AbstractBaseDomain<OpProductVO, OpProduct> implements OpProductDomain {

    private static final Logger LOGGER = LoggerFactory.getLogger(OpProductDomainImpl.class);
    private static final String DEFAULT_CONBINED_SPV_SKU_CODE = "__";
    private static final int DEFAULT_SPV_MIN_AMOUNT = 1;

    @Autowired
    private OpProductMapper opProductMapper;
    @Autowired
    private OpProdSkuDomain opProdSkuDomain;
    @Autowired
    private PcsSkuDomain pcsSkuDomain;
    @Autowired
    private OpChnCanSaleProdDomain opChnCanSaleProdDomain;
    @Autowired
    private OpProdCanSelfDeliveryDomain opProdCanSelfDeliveryDomain;
    @Autowired
    private OpChannelDomain opChannelDomain;
    @Autowired
    private OpProdDynmContDomain opProdDynmContDomain;
    @Autowired
    private OpAttributeDomain opAttributeDomain;
    @Autowired
    private OpSpvPriceChangeDomain opSpvPriceChangeDomain;
    @Autowired
    private PcsSkuCombinationDomain pcsSkuCombinationDomain;
    @Autowired
    private PropertyConfigurer propertyConfigurer;


    private Map<String, Object> mapping = new HashMap<>();
    {
        mapping.put("product.delivery.fedex.categories", "119,305");
        mapping.put("product.delivery.sfair.categories", "120");
    }
//    @Autowired
//    private PropertyConfigurer propertyConfigurer;

    /**
     * 从Properties中获取类目列表
     * @author gongjun
     * @return
     */
    private List<Long> getCategoryIdsFromProperties(String propertiesKey) {
//        String strResult = propertyConfigurer.getProperty("product.delivery.fedex.categories");
        String strResult = (String) mapping.get(propertiesKey);
        if (StringUtils.isBlank(strResult)) {
            return Lists.newArrayList();
        }
        String[] values = strResult.split(",");
        List<Long> results = Lists.newArrayList();
        for (String value : values) {
            results.add(Long.valueOf(value));
        }
        return results;
    }

    /**
     * 获取默认联邦快递的默认类目列表
     * @author gongjun
     * @return
     */
    private List<Long> getFexExCategoryIds() {
        return getCategoryIdsFromProperties("product.delivery.fedex.categories");
    }

    /**
     * 获取默认顺风空运的默认类目列表
     * @author gongjun
     * @return
     */
    private List<Long> getSFAirCategoryIds() {
        return getCategoryIdsFromProperties("product.delivery.sfair.categories");
    }


    /**
     * 设置默认的配送方式
     * @author gongjun
     * @param product
     * @param skuCodes
     */
    private void setupDefaultExpressType(final OpProduct product, List<String> skuCodes) {
        // 配置方式为联邦的适用类目编号集合
        List<Integer> fxCategoryIds = getCategoriesByType("prod.delivery.FX.categories");
        // 配送方式为顺丰空运的适用类目编号集合
        List<Integer> sfaCategoryIds = getCategoriesByType("prod.delivery.SFA.categories");
        LOGGER.info("[设置默认配送方式] 联邦类目: {}, 顺风空运类目: {}", StringUtils.join(fxCategoryIds, ", "),
                StringUtils.join(sfaCategoryIds, ", "));
        // 商品下的sku有为鲜花或者鲜花花材的二级类目，则设置配送方式为联邦
        // 只存在永生花的二级类目的配送方式为顺丰空运否则设置为顺丰陆运
        if (CollectionUtils.isNotEmpty(fxCategoryIds)
                && pcsSkuDomain.isExistsSpecialCategory(skuCodes, fxCategoryIds)) {
            LOGGER.info("[设置默认配送方式:联邦]");
            product.setExpressType(Integer.valueOf(propertyConfigurer.getProperty("prod.delivery.type.FX", "0")));
        } else if(org.apache.commons.collections4.CollectionUtils.isNotEmpty(sfaCategoryIds)
                && pcsSkuDomain.isOnlyExistsSpecialCategory(skuCodes, sfaCategoryIds)){
            LOGGER.info("[设置默认配送方式:顺风空运]");
            product.setExpressType(Integer.valueOf(propertyConfigurer.getProperty("prod.delivery.type.SFA", "0")));
        }else {
            LOGGER.info("[设置默认配送方式:顺风陆运]");
            product.setExpressType(Integer.valueOf(propertyConfigurer.getProperty("prod.delivery.type.default", "0")));
        }
    }

    /**
     * 获取配置文件中配置的类目
     * @param type
     * @return
     */
    private List<Integer> getCategoriesByType(String type) {
        List<Integer> categoryIds = Lists.newArrayList();
        String[] split = propertyConfigurer.getProperty(type, "0").split(",");
        if(null != split && split.length > 0) {
            try{
                for (String s : split) {
                    categoryIds.add(Integer.valueOf(s));
                }
            }catch (NumberFormatException e){
                LOGGER.error("配置文件中配送快递默认二级类目编号指定非法");
                throw new OperationException("配置文件中配送快递默认二级类目编号指定非法");
            }
        }
        LOGGER.info("配置的二级类目编号为：" + categoryIds);
        return categoryIds;
    }


    private boolean createWithSkuCodes(final OpProductVO prod, final List<String> skuCodes, boolean createSku) {
        final OpProduct product = BeanUtil.buildFrom(prod, OpProduct.class);
        // 设置默认配送方式
        setupDefaultExpressType(product, skuCodes);
        opProductMapper.insert(product);
        final Map<String, Object> params = new HashMap<>();
        final Long prodId = product.getId();
        params.put("id", prodId);
        product.setCode(CodeGenerator.getInstance().generate("OP_PROD_CODE", params));
        if (createSku && CollectionUtils.isNotEmpty(skuCodes)) {
            for (final String skuCode : skuCodes) {
                final OpProdSku vo = new OpProdSku();
                vo.setProductId(prodId);
                vo.setSkuCode(skuCode);
                // opProdSkuMapper.insert(vo);
                opProdSkuDomain.create(vo);
            }
        }
        opProductMapper.updateByPrimaryKeySelective(product);
        prod.setId(product.getId());
        return product.getId() > 0;
    }

    /**
     * 创建Product
     * @param prod
     * @return
     */
    @Override
    public boolean create(final OpProductVO prod) {
        final List<String> skuCodes = prod.getSkuCodes();
        return createWithSkuCodes(prod, skuCodes, true);
    }


    private List<String> getCombinedSkuCodes(OpProductVO prod) {
        List<String> skuCodes = new ArrayList<>();
        List<OpSpvVO> spvList = prod.getSpvList();
        if (CollectionUtils.isNotEmpty(spvList)) {
            for (OpSpvVO spvVO : spvList) {
                List<PcsSkuCombinationVO> skuCmbList = spvVO.getSkuCombinationList();
                for (PcsSkuCombinationVO skuCmbVO : skuCmbList) {
                    skuCodes.add(skuCmbVO.getSubSkuCode());
                }
            }
        }
        return skuCodes;
    }

    @Override
    public boolean createCombinedProduct(OpProductVO prod) {
        prod.setCombined(true);
        List<String> skuCodes = getCombinedSkuCodes(prod);
        if (!createWithSkuCodes(prod, skuCodes, false)) {
            return false;
        }
        List<OpAttributeVO> attributeList = prod.getAttributeList();
        Map<String, OpAttributeVO> attrIdMap = new LinkedHashMap<>();
        if (CollectionUtils.isNotEmpty(attributeList)) {
            for (OpAttributeVO attrVo : attributeList) {
                attrVo.setProductId(prod.getId());
                attrVo.setGroupName(attrVo.getName());
                attrVo.setCombined(1);
                opAttributeDomain.createWithValues(attrVo);
                attrIdMap.put(attrVo.getName(), attrVo);
            }
        }
        List<OpSpvVO> spvList = prod.getSpvList();
        Integer prodStatus = prod.getProdStatus();
        if (CollectionUtils.isNotEmpty(spvList)) {
            for (OpSpvVO spvVo : spvList) {
                spvVo.setProductId(prod.getId());
                spvVo.setSkuCode(DEFAULT_CONBINED_SPV_SKU_CODE);
                spvVo.setMinAmount(DEFAULT_SPV_MIN_AMOUNT);
                opProdSkuDomain.createCombinedSpv(attrIdMap, spvVo, prodStatus);
            }
        }
        return true;
    }

    @Override
    public boolean updateCombinedProduct(OpProductVO prod) {
        //更新商品
        prod.setCombined(true);
        OpProduct product = new OpProduct();

        product.setId(prod.getId());
        product.setName(prod.getName());
        product.setNameCn(prod.getNameCn());
        product.setDescription(prod.getDescription());
        product.setProdStatus(prod.getProdStatus());
        product.setIsLuckybag(prod.getIsLuckybag());

        int temProdStatus = prod.getTemProdStatus();
        int prodStatus = prod.getProdStatus();

//        int flag = opProductMapper.updateByPrimaryKeySelective(product);
//        if( flag < 1 ){
//            return false;
//        }
        //更新t_op_attribute、t_op_attribute_value
        List<OpAttributeVO> attributeList = prod.getAttributeList();

        if (CollectionUtils.isNotEmpty(attributeList)) {
            for(int i=0;i<attributeList.size();i++){
                OpAttributeVO attrVo = attributeList.get(i);
                attrVo.setId(attrVo.getId());
                attrVo.setGroupName(attrVo.getName());
                boolean flag2 = opAttributeDomain.updateWithValues(attrVo);
                if(!flag2) return flag2;
            }

        }
        List<OpSpvVO> spvList = prod.getSpvList();

        boolean flag4 = false;

        if (CollectionUtils.isNotEmpty(spvList)) {
            for(int i=0;i<spvList.size();i++){
                OpSpvVO spvVo = spvList.get(i);
                OpSpvPriceChange obj = opSpvPriceChangeDomain.getCurrentAduitPrice(spvVo.getId());
                if(obj != null) {
                    BigDecimal salesPrice = obj.getSalesPrice();
                    if (salesPrice.longValue() != spvVo.getSalesPrice().longValue()) {
                        flag4 = true;
                    }
                }
                boolean flag3 =  opProdSkuDomain.updateCombinedSpv(spvVo, temProdStatus, prodStatus);
                if(!flag3) return flag3;
            }
        }
        if(!flag4 && prod.getTemProdStatus().longValue() == 9){
            product.setProdStatus(prod.getTemProdStatus());
        }

        int flag = opProductMapper.updateByPrimaryKeySelective(product);
        if( flag < 1 ){
            return false;
        }

        return true;
    }

    @Override
    public boolean update(final OpProductVO product) {
        return opProductMapper.updateByPrimaryKeySelective(BeanUtil.buildFrom(product, OpProduct.class)) != 0;
    }

    @Override
    public boolean updateProdAndSku(final OpProductVO product) {
        opProdSkuDomain.deleteByProdId(product.getId());
        if (product.getSkuCodes() != null && !product.getSkuCodes().isEmpty()) {
            for (String skuCode : product.getSkuCodes()) {
                OpProdSku prodSku = new OpProdSku();
                prodSku.setSkuCode(skuCode);
                prodSku.setProductId(product.getId());
                opProdSkuDomain.create(prodSku);
            }
        }
        OpProduct opProduct = BeanUtil.buildFrom(product, OpProduct.class);
        return opProductMapper.updateByPrimaryKeySelective(opProduct) != 0;
    }

    public boolean updateN(final OpProductVO product) {
        final Long prodId = product.getId();
        final List<String> skuCodes = product.getSkuCodes();
        if (CollectionUtils.isEmpty(skuCodes)) {
            return opProductMapper.updateByPrimaryKeySelective(BeanUtil.buildFrom(product, OpProduct.class)) != 0;
        }
        final List<OpProdSku> skus = opProdSkuDomain.findByProdId(prodId);
        final List<String> existSkus = new ArrayList<>(skus.size());
        final Map<String, Long> existSkuIds = new HashMap<>();
        for (final OpProdSku sku : skus) {
            existSkus.add(sku.getSkuCode());
            existSkuIds.put(sku.getSkuCode(), sku.getId());
        }
        final List<String> delSkus = new ArrayList<>(existSkus);
        final List<String> newSkus = new ArrayList<>(skuCodes);
        delSkus.removeAll(skuCodes);
        newSkus.removeAll(existSkus);

        if (CollectionUtils.isNotEmpty(newSkus)) {
            for (final String newSku : newSkus) {
                final OpProdSku vo = new OpProdSku();
                vo.setProductId(prodId);
                vo.setSkuCode(newSku);
                // opProdSkuMapper.insert(vo);
                opProdSkuDomain.create(vo);
            }
        }
        if (CollectionUtils.isNotEmpty(delSkus)) {
            for (final String delSku : delSkus) {
                opProdSkuDomain.deleteById(existSkuIds.get(delSku));
            }
            final List<OpChnCanSaleProdVO> opChnCanSaleProds = opChnCanSaleProdDomain
                    .findByProdIds(Arrays.asList(prodId));
            if (CollectionUtils.isNotEmpty(opChnCanSaleProds)) {
                final List<String> allSkus = new ArrayList<>(newSkus);
                allSkus.addAll(existSkus);
                allSkus.removeAll(delSkus);
                for (final OpChnCanSaleProdVO opChnCanSaleProd : opChnCanSaleProds) {
                    final OpChnCanSaleProdVO updateCanSalProd = new OpChnCanSaleProdVO();
                    updateCanSalProd.setId(opChnCanSaleProd.getId());
                    final List<String> saleSkus = new ArrayList<>(
                            Arrays.asList(opChnCanSaleProd.getCanSaleSku().split(",")));
                    final List<String> seeSkus = new ArrayList<>(
                            Arrays.asList(opChnCanSaleProd.getCanSeeSku().split(",")));
                    saleSkus.retainAll(allSkus);
                    seeSkus.retainAll(allSkus);
                    if (CollectionUtils.isNotEmpty(saleSkus)) {
                        updateCanSalProd.setCanSaleSku(StringUtils.join(saleSkus, ","));
                    }
                    if (CollectionUtils.isNotEmpty(seeSkus)) {
                        updateCanSalProd.setCanSeeSku(StringUtils.join(seeSkus, ","));
                    }
                    opChnCanSaleProdDomain.update(BeanUtil.buildFrom(updateCanSalProd, OpChnCanSaleProd.class));
                }
            }
        }
        return opProductMapper.updateByPrimaryKeySelective(BeanUtil.buildFrom(product, OpProduct.class)) != 0;
    }

    @Override
    public boolean deleteById(final Long id) {
        // 判断是否各渠道是否已下架prod
        final List<OpChnCanSaleProdVO> chnCanSaleProds = opChnCanSaleProdDomain
                .findByProdIds(Collections.singletonList(id));
        final List<String> onshellChnCodes = new ArrayList<>();
        final int i = 0;
        for (final OpChnCanSaleProdVO chnCanSaleProd : chnCanSaleProds) {
            if (PegasusConstants.STATUS_AVAILABLE == chnCanSaleProd.getOnShelf()) {
                final OpChannel channel = opChannelDomain.findByCode(chnCanSaleProd.getChannelCode());
                if (channel != null) {
                    onshellChnCodes.add(channel.getName());
                }
            }
        }
        if (CollectionUtils.isNotEmpty(onshellChnCodes)) {
            throw new OperationException(OperationExceptionErrorCode.CHANNEL_DELETE_CANSALESKU_ERROR,
                    "请先把" + StringUtils.join(onshellChnCodes, ",") + "下架，再删除！");
        }
        try {
            opProdSkuDomain.deleteByProdId(id);
            opChnCanSaleProdDomain.deleteByProdId(id);
            opProdCanSelfDeliveryDomain.deleteByProdId(id);
            opProdDynmContDomain.deleteByProdId(id);
            return opProductMapper.deleteByPrimaryKey(id) != 0;
        } catch (final Exception e) {
            log.error("", e);
            throw new OperationException(OperationExceptionErrorCode.CHANNEL_DELETE_CANSALESKU_ERROR, "删除PROD失败！");
        }
    }

    @Override
    public List<OpProductVO> findByCond(final OpProductCond cond) {
        List<OpProductVO> list = BeanUtil.buildListFrom(opProductMapper.findByCriteria(cond), OpProductVO.class);
        for (OpProductVO vo : list) {
            if (cond.getWithAttribute() != null && cond.getWithAttribute()) {
                OpAttributeCond attrCond = new OpAttributeCond();
                attrCond.setProductId(vo.getId());
                attrCond.setWithValues(true);
                attrCond.setActive(true);
                List<OpAttributeVO> attrList = opAttributeDomain.findByCondition(attrCond);
                vo.setAttributeList(attrList);
            }
            if (cond.getWithSpv() != null && cond.getWithSpv()) {
                OpSpvCond spvCond = new OpSpvCond();
                spvCond.setProductId(vo.getId());
                spvCond.setWithSku(cond.getWithSku());
                spvCond.setWithSkuCombination(cond.getWithSku());
                List<OpSpvVO> spvList = opProdSkuDomain.findByPcsProdSkuCond(spvCond);
                vo.setSpvList(spvList);

                if (vo.getCombined() != null && vo.getCombined()) {
                    Map<String, OpSubSkuGroupVO> subSkuGroups = getSubSkuGroups(vo);
                    vo.setSubSkuGroups(subSkuGroups);
                }
            }
        }
        return list;
    }


    private Map<String, OpSubSkuGroupVO> getSubSkuGroups(OpProductVO productVO) {
        Map<String, OpSubSkuGroupVO> groups = new LinkedHashMap<>();
        List<OpSpvVO> spvList = productVO.getSpvList();
        List<OpAttributeVO> attributeList = productVO.getAttributeList();
        Map<Long, PcsSkuCombinationVO> attrValueIdSubSkuMap = new HashMap<>();
        for (OpSpvVO spvVO : spvList) {
            List<PcsSkuCombinationVO> skuCombinationList = spvVO.getSkuCombinationList();
            for (PcsSkuCombinationVO skuCombinationVO : skuCombinationList) {
                LinkedHashMap<Long, Long> subSkuAttrMap = skuCombinationVO.getAttrValueIds();
                for (Iterator<Long> idIter = subSkuAttrMap.keySet().iterator(); idIter.hasNext(); ) {
                    Long id = idIter.next();
                    attrValueIdSubSkuMap.put(subSkuAttrMap.get(id), skuCombinationVO);
                }
            }
        }
        for (OpAttributeVO attrVO : attributeList) {
            List<OpAttributeValueVO> attrValueList = attrVO.getAttributeValueList();
            OpSubSkuGroupVO subSkuGroupVO = groups.get(attrVO.getGroupName());
            if (subSkuGroupVO == null) {
                subSkuGroupVO = new OpSubSkuGroupVO(attrVO.getGroupName());
                subSkuGroupVO.setAttributeId(attrVO.getId());
                groups.put(attrVO.getGroupName(), subSkuGroupVO);
            }
            for (OpAttributeValueVO attrValueVO : attrValueList) {
                PcsSkuCombinationVO subSkuVO = attrValueIdSubSkuMap.get(attrValueVO.getId());
                subSkuVO.setAttrName(attrVO.getName());
                subSkuVO.setAttrValue(attrValueVO.getValueText());
                subSkuVO.setOpAttributeVO(attrVO);
                subSkuGroupVO.addSubSku(subSkuVO);
            }
        }
        return groups;
    }



    @Override
    public List<OpProductVO> findBySkuCode(String skuCode) {
        return BeanUtil.buildListFrom(opProductMapper.findBySkuCode(skuCode), OpProductVO.class);
    }

    @Override
    public List<OpProductExcelVO> findExportListByCond(final OpProductCond cond) {
        return BeanUtil.buildListFrom(opProductMapper.findExportListByCriteria(cond), OpProductExcelVO.class);
    }

    @Override
    public List<OpProductVO> findByNameOrCode(final String name) {
        final OpProductExample example = new OpProductExample();
        final OpProductExample.Criteria criteria = example.createCriteria();
        criteria.andCodeLike(SQLUtils.allLike(name));
        example.or().andNameLike(SQLUtils.allLike(name));
        example.or().andNameCnLike(SQLUtils.allLike(name));
        example.setOrderByClause("CODE  LIMIT 20");
        return BeanUtil.buildListFrom(opProductMapper.selectByExample(example), OpProductVO.class);
    }

    @Override
    public List<OpProductVO> findProdWithSpvByNameOrCode(String arg0){
        List<OpProductVO> prods = BeanUtil.buildListFrom(opProductMapper.findNoCombinedProdByCodeOrName(arg0), OpProductVO.class);
        for(OpProductVO prod : prods){
            OpSpvCond spvCond = new OpSpvCond();
            spvCond.setProductId(prod.getId());
            spvCond.setWithSku(false);
            List<String> skuCodes = new ArrayList<String>();
            List<OpProdSku> spvList = opProdSkuDomain.findByProdId(prod.getId());
            //prod.setSpvList(spvList);
            for (OpProdSku skuVO : spvList){
                skuCodes.add(skuVO.getSkuCode());
            }
            prod.setSkuCodes(skuCodes);
        }
        return prods;
    }

    @Override
    public OpProductVO findById(final Long id) {
        OpProduct opProduct = opProductMapper.selectByPrimaryKey(id);
        return BeanUtil.buildFrom(opProduct, OpProductVO.class);
    }

    @Override
    public List<OpProductVO> findByIds(final List<Long> ids) {
        return findByIds(ids, false);
    }

    @Override
    public List<OpProductVO> findProdWithWatermarkByIds(final List<Long> ids) {
        return findProdWithWatermarkByIds(ids, false);
    }

    @Override
    public List<OpProductVO> findProdWithTagByIds(final List<Long> ids) {
        return findProdWithTagByIds(ids, false);
    }

    @Override
    public List<Long> findProdWatermarkProdIdsByParams(final Map<String, Object> params) {
        return opProductMapper.findProdWatermarkProdIdsByParams(params);
    }

    @Override
    public List<Long> findProdTagProdIdsByParams(final Map<String, Object> params) {
        return opProductMapper.findProdTagProdIdsByParams(params);
    }

    @Override
    public List<OpProduct> listProdMat(Map<String, Object> params) {
        Integer page = MapUtils.getInteger(params, "page", Integer.valueOf(1));
        Integer size = MapUtils.getInteger(params, "size", Integer.valueOf(20));
        if(page.intValue() > 0) {
            page = Integer.valueOf(page.intValue() - 1);
        }

        int start = page.intValue() * size.intValue();
        params.put("start", Integer.valueOf(start));
        params.put("criteriaStr", "LIMIT " + start + ", " + size);
        if(params.get("combined") == null){
            params.put("combined","-1");
        }
        List<OpProduct> list = opProductMapper.listProdMat(params);
        log.info("[listProdMat] result: " + JSON.toJSONString(list));
        return list;
    }


    public List<OpProductVO> findProdByIds(List<Long> ids, boolean withCategories) {
        final Map<String, Object> params = new HashMap<>();
        params.put("ids", ids);
        List<OpProduct> productList = opProductMapper.queryByIds(params);
        List<OpProductVO> productVOList = BeanUtil.buildListFrom(productList, OpProductVO.class);
        return productVOList;
    }

    @Override
    public List<OpWatermarkVO> queryWatermarksByProdId(final Map<String,Object> params) {
        List<OpWatermarkVO> watermarkList = opProductMapper.queryWatermarksByParams(params);
        return watermarkList;
    }

    @Override
    public List<OpTagVO> queryTagsByProdId(final Map<String,Object> params) {
        List<OpTagVO> tagList = opProductMapper.queryTagsByParams(params);
        return tagList;
    }

    @Override
    public List<Map> listProdMat(OpProductCond cond) {
//        List<Map> list = opProductMapper.listProdMat(cond);
//        log.info("[listProdMat] result: " + JSON.toJSONString(list));
//        return list;
        return null;
    }

    public List<OpProductVO> findProdWithWatermarkByIds(List<Long> ids, boolean withCategories) {
        // OpProductExample example = new OpProductExample();
        // OpProductExample.Criteria criteria = example.createCriteria();
        // criteria.andIdIn(ids);
        // return
        // BeanUtil.buildListFrom(opProductMapper.selectByExample(example),OpProductVO.class);
        List<OpProductVO> productVOList = findProdByIds(ids, withCategories);
        Map<String,Object> params = new HashMap<String,Object>();
        for (OpProductVO opProd : productVOList) {
            params.put("id", opProd.getId());
            params.put("active", 1);
            List<OpWatermarkVO> watermarkList = opProductMapper.queryWatermarksByParams(params);
            opProd.setWatermarkLis(watermarkList);
        }
        return productVOList;
    }

    public List<OpProductVO> findProdWithTagByIds(List<Long> ids, boolean withCategories) {
        List<OpProductVO> productVOList = findProdByIds(ids, withCategories);
        Map<String,Object> params = new HashMap<String,Object>();
        for (OpProductVO opProd : productVOList) {
            params.put("id", opProd.getId());
            params.put("active", 1);
            List<OpTagVO> tagList = opProductMapper.queryTagsByParams(params);
            opProd.setTags(tagList);
        }
        return productVOList;
    }

    @Override
    public List<OpProductVO> findByIds(List<Long> ids, boolean withCategories) {
        // OpProductExample example = new OpProductExample();
        // OpProductExample.Criteria criteria = example.createCriteria();
        // criteria.andIdIn(ids);
        // return
        // BeanUtil.buildListFrom(opProductMapper.selectByExample(example),OpProductVO.class);
        final Map<String, Object> params = new HashMap<>();
        params.put("ids", ids);
        List<OpProduct> productList = opProductMapper.queryByIds(params);
        List<OpProductVO> productVOList = BeanUtil.buildListFrom(productList, OpProductVO.class);

        return productVOList;
    }

    @Override
    public List<Long> findProductIdsByCampaignIds(List<Long> ids) {
        final Map<String, Object> params = new HashMap<>();
        params.put("ids", ids);
        List<Long> idList = opProductMapper.findProductIdsByCampaignIds(params);
        return idList;
    }

    @Override
    public List<Long> findProductIdsByCodes(List<String> codes) {
        final Map<String, Object> params = new HashMap<>();
        params.put("codes", codes);
        List<Long> idList = opProductMapper.findProductIdsByCodes(params);
        return idList;
    }

    @Override
    public int updateByExampleSelective(final OpProductVO vo) {
        final OpProductExample example = new OpProductExample();
        final OpProductExample.Criteria criteria = example.createCriteria();
        criteria.andCodeEqualTo(vo.getCode());
        final OpProduct product = new OpProduct();
        product.setSeq(vo.getSeq());
        product.setSort1(vo.getSort1());
        product.setSort2(vo.getSort2());
        return opProductMapper.updateByExampleSelective(product, example);
    }


    @Override
    public OpProductVO findProductByCode(String code) {
//        OpProductVO opProductVO = new OpProductVO();
        OpProduct po = opProductMapper.selectByCode(code);
        OpProductVO opProductVO = BeanUtil.buildFrom(po, OpProductVO.class);
//        if (opProduct != null && opProduct.getId() != null) {
//            opProductVO.setId(opProduct.getId());
//            opProductVO.setCategory1Id(opProduct.getCategory1Id());
//            opProductVO.setCategory1Name(opProduct.getCategory1Name());
//            opProductVO.setCategory2Id(opProduct.getCategory2Id());
//            opProductVO.setCategory2Name(opProduct.getCategory2Name());
//            opProductVO.setFirstOnSaleTime(opProduct.getFirstOnSaleTime());
//            opProductVO.setCode(opProduct.getCode());
//            opProductVO.setListPrice(opProduct.getListPrice());
//            opProductVO.setAllowVipDiscount(opProduct.getAllowVipDiscount());
//            opProductVO.setAllowNoteCard(opProduct.getAllowNoteCard());
//            opProductVO.setPrepareDays(opProduct.getPrepareDays());
//            opProductVO.setSelectDeliveryDays(opProduct.getSelectDeliveryDays());
//            opProductVO.setPicList(opProduct.getPicList());
//            opProductVO.setPicDetail(opProduct.getPicDetail());
//            opProductVO.setPicDetailMul(opProduct.getPicDetailMul());
//            opProductVO.setPicThumb(opProduct.getPicThumb());
//            opProductVO.setName(opProduct.getName());
//            opProductVO.setNameCn(opProduct.getNameCn());
//            opProductVO.setSkuCategoryDesc(opProduct.getSkuCategoryDesc());
//            opProductVO.setExpressType(opProduct.getExpressType());
//            opProductVO.setSaleStartDate(opProduct.getSaleStartDate());
//            opProductVO.setSort1(opProduct.getSort1());
//            opProductVO.setSort2(opProduct.getSort2());
//            opProductVO.setSeq(opProduct.getSeq());
//            opProductVO.setCombined(opProduct.getCombined());
//        }
        return opProductVO;
    }

    @Override
    public Boolean isExistsSpecialCategory(Long productId, List<Integer> categoryIds) {
        return opProductMapper.getSpecialCategoryByProductId(productId, categoryIds) > 0;
    }

    @Override
    public Boolean isOnlyExistsSpecialCategory(Long productId, List<Integer> categoryIds) {
        // 是否只存在指定的二级类目，判断方法为：存在指定类目并且不存在指定类目以外的类目
        return opProductMapper.getSpecialCategoryByProductId(productId, categoryIds) > 0
                && opProductMapper.getOtherCategoryByProductId(productId, categoryIds) <= 0;
    }

    @Override
    public Long getIdByCode(String code) {
        if(code == null) {
            return null;
        }
        return opProductMapper.getIdByCode(code);
    }

    @Override
    public String getCodeById(Long id) {
        if(id == null) {
            return null;
        }
        OpProduct opProduct = opProductMapper.selectByPrimaryKey(id);
        if(opProduct  == null) {
            return null;
        }
        return opProduct.getCode();
    }

    @Override
    public OpProductVO findCombinedProductByProdId(Long productId){
        OpProductVO opProductVO = BeanUtil.buildFrom(opProductMapper.selectByPrimaryKey(productId),OpProductVO.class);
        if(null != opProductVO && opProductVO.getCombined()){
            OpSpvCond spvCond = new OpSpvCond();
            spvCond.setProductId(productId);
            spvCond.setWithSkuCombination(true);
            opProductVO.setSpvList(opProdSkuDomain.findSpvCombinedByProdId(productId));
        }
        return opProductVO;
    }

    @Override
    public List<OpProductVO> findCombinedProdAuditList(OpProductCond cond){
        return opProductMapper.findCombinedProdAuditList(cond);
    }

    @Override
    public Boolean auditCombinedProdByProdId(OpProductCond cond){
        if(null == cond.getProdStatus()){
            throw new OperationException(OperationExceptionErrorCode.ILLEGAL_PARAM,
                    "产品审核状态不能为空！");
        }
        if (null == cond.getChangeStatus()){
            throw new OperationException(OperationExceptionErrorCode.ILLEGAL_PARAM,
                    "价格变更状态不能为空！");
        }
        if(null == cond.getId()){
            throw new OperationException(OperationExceptionErrorCode.ILLEGAL_PARAM,
                    "产品Id不能为空！");
        }

        OpProduct baseicOpProduct = opProductMapper.selectByPrimaryKey(cond.getId());
        int temProdStatus = baseicOpProduct.getProdStatus();

        OpProduct opProduct = new OpProduct();
        opProduct.setId(cond.getId());
        opProduct.setProdStatus(cond.getProdStatus());
        opProduct.setLastAuditTime(new Date());
        opProductMapper.updateByPrimaryKeySelective(opProduct);
        opSpvPriceChangeDomain.updatePriceChangeStatusByProductId(cond.getId(), cond.getChangeStatus());
        //如果当前商品是财务审批中 点击审批通过 则更新SPV表价格
        if(temProdStatus == 3 && cond.getProdStatus() == 9) {
            List<OpProdSku> list = opProdSkuDomain.findByProdId(cond.getId());
            if (CollectionUtils.isNotEmpty(list)) {
                for (OpProdSku spv : list) {
                    Long spvId = spv.getId();
                    OpSpvPriceChange obj = opSpvPriceChangeDomain.getCurrentAduitPrice(spvId);
                    BigDecimal price = obj.getSalesPrice();
                    OpProdSku opProdSku = new OpProdSku();
                    opProdSku.setId(spvId);
                    opProdSku.setSalesPrice(price);
                    opProdSku.setCombined(true);
                    opProdSkuDomain.update(opProdSku);
                }
            }
        }
        return true;
    }

    @Override
    public Boolean getCrossBorderFlagListByCode(String code, Integer crossBorderFlag) {
        return opProductMapper.getCrossBorderFlagListByCode(code, crossBorderFlag) > 0;
    }

    @Override
    public List<PcsSkuVO> getSkuListByProductCode(String productCode) {
        OpProductVO productVO = this.findProductByCode(productCode);
        // 绑定商品下SKU所对应的类目ID
        List<String> skuCodes = Lists.newArrayList();
        if(productVO.getCombined()){
            // 组合商品 默认取 Product下第一个Spv 下第一个Sku
            OpProductVO product = findCombinedProductByProdId(productVO.getId());
            List<OpSpvVO> spvList = product.getSpvList();
            if(product != null && CollectionUtils.isNotEmpty(spvList)){
                for (OpSpvVO opSpvVO : spvList) {
                    List<PcsSkuCombinationVO> list = opSpvVO.getSkuCombinationList();
                    if (CollectionUtils.isNotEmpty(list)) {
                        for (PcsSkuCombinationVO item : list) {
                            skuCodes.add(item.getSubSkuCode());
                        }
                    }
                }
            }
        }else {
            // 普通商品
            List<OpProductVO> productList = findProdWithSpvByNameOrCode(productCode);
            if(CollectionUtils.isNotEmpty(productList)) {
                for(OpProductVO opProductVO : productList) {
                    skuCodes.addAll(opProductVO.getSkuCodes());
                }
            }
        }
        List<PcsSkuVO> resultList = pcsSkuDomain.findByCodes(skuCodes);
        return resultList;
    }
}