package com.thebeastshop.pegasus.component.price;

import java.math.BigDecimal;
import java.util.Map;

import org.springframework.core.Ordered;

import com.thebeastshop.cart.ProductPack;;

/**
 * 计价器
 * 
 * <pre>
 * 将有多个实现类：会员折扣、活动折扣、礼品卡、优惠券、优惠码、以及可能会出现的“下单来源折扣”（比如要推app，搞“手机下单优惠xxx”）等等。
 * 需要进行价格计算的地方（购物车展示、订单确认页面、下单结算）导入所有实现类，依次计算价格。前一个实现类的价格计算结果将影响后一个的计算结果，所以实现类列表是有序的，其顺序依照产品设计的优惠顺序。
 * 注：上面的“原价”是相对原价。比如说有多个价格计算服务，则后面得到的“原价”是前面输出的结果价，不是商品最开始是原价。
 * 附目前需求结算顺序：会员折扣>促销活动＞优惠券
 * 要展示商品价格的地方：购物车、订单确认页面
 *
 * </pre>
 * 
 * 通常的用法：
 * 
 * <pre>
 * \@Autowired
 * private List< PriceCalculator > list;
 *
 * ......
 *
 * List< ProductPack> packs=...
 * //初始化最开始的价格关系
 * Map< ProductPack, Double > rawPriceMap=new ...
 * for(ProductPack p:packs){
 *     rawPriceMap.put(p,p.getFactProductPrice());
 * }
 * //创建上下文对象
 * PriceCalculateContextForProduct c=new PriceCalculateContextForProduct();
 * c.setPackAndOriginalPrice(rawPriceMap);
 * c.set...//设置其他该设置的属性
 * for(PriceCalculator pc:list){
 *     Map< ProductPack, Double > finalPriceMap=pc.calculate(c);
 *     c.setPackAndOriginalPrice(finalPriceMap);
 * }
 * </pre>
 *
 * @author Liang Wenjian
 */
public interface PriceCalculator extends Ordered {

	/**
	 * <pre>
	 * 计算优惠后的价格。
	 * 注：
	 * 若有一些商品包没有参与折扣，返回的map中也包含该商品包和其原价的映射。即返回的map的size应=输入的商品包列表的size。
	 * <b>不</b>检查使用的合法性！！！请调用处自行检查。
	 * </pre>
	 *
	 * @param context
	 * @return
	 */
	Map<ProductPack, BigDecimal> calculate(final PriceCalculateContext context);

}
