package com.thebeastshop.pegasus.component.discount;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.thebeastshop.pegasus.component.campaign.CampaignSection;
import com.thebeastshop.pegasus.component.campaign.FactorTypeEnum;
import com.thebeastshop.cart.ProductPack;
import com.thebeastshop.support.exception.BusinessLogicNotExpectedException;
import com.thebeastshop.support.mark.HasIdGetter.HasIntIdGetter;
import com.thebeastshop.support.mark.HasName;
import com.thebeastshop.support.util.PriceUtil;

/**
 * 优惠类型
 *
 * @author Liang Wenjian
 */
public enum DiscountType implements HasIntIdGetter, HasName {
	/**
	 * 价格折扣
	 */
	OFF(1, "价格折扣", 1) {
		@Override
		public boolean isAffectPrice() {
			return true;
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section) {
			final FactorTypeEnum factorType = section.getFactorType();
			if (null != factorType) {
				return factorType.calculate(price, section);
			}
			throw new BusinessLogicNotExpectedException("价格影响因子类型错误");
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section, final int times) {
			// 打折还打好几次？少来。
			return calculatePrice(price, section);
		}

		@Override
		public BigDecimal handlePrice(final List<ProductPack> participatingPacks, final CampaignSection campaignSection,
				final BigDecimal participatingPrice, final int times, final Map<ProductPack, BigDecimal> finalPriceMap,
				final Map<ProductPack, BigDecimal> preFinalPriceMap) {
			BigDecimal participatedPrice = buildParticipatingPack(participatingPacks, campaignSection, times,
					finalPriceMap, preFinalPriceMap);
			return PriceUtil.keepToCent(participatedPrice);
		}
	},
	/**
	 * 满减
	 */
	CHEAPEN(2, "满减", 2) {
		@Override
		public boolean isAffectPrice() {
			return true;
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section) {
			return price.subtract(section.getFactor());
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section, final int times) {
			return PriceUtil.keepToCent(price.subtract(section.getFactor().multiply(BigDecimal.valueOf(times))));
		}

		@Override
		public BigDecimal handlePrice(final List<ProductPack> participatingPacks, final CampaignSection campaignSection,
				final BigDecimal participatingPrice, final int times, final Map<ProductPack, BigDecimal> finalPriceMap,
				final Map<ProductPack, BigDecimal> preFinalPriceMap) {
			final BigDecimal participatedPrice = calculatePrice(participatingPrice, campaignSection, times);
			// 每一份钱省下的比例。
			final BigDecimal savedRate = participatingPrice.subtract(participatedPrice).divide(participatingPrice, 10,
					RoundingMode.HALF_UP);
			for (final ProductPack participatingPack : participatingPacks) {
				BigDecimal rawProductPrice = preFinalPriceMap.get(participatingPack);
				if (rawProductPrice == null) {
					rawProductPrice = participatingPack.getFactProductPrice();
				}
				final BigDecimal participatedProductPrice = PriceUtil
						.keepToCent(rawProductPrice.subtract(rawProductPrice.multiply(savedRate)));
				finalPriceMap.put(participatingPack, participatedProductPrice);
			}
			return participatedPrice;
		}
	},
	/**
	 * 满折
	 */
	DIC(5, "满折", 2) {
		@Override
		public boolean isAffectPrice() {
			return true;
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section) {
			// 打折计算规格：price * (factor / 10)
			return PriceUtil.keepToCent(
					price.multiply(section.getFactor().divide(BigDecimal.valueOf(10), 2, RoundingMode.HALF_UP)));
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section, final int times) {
			// 打折还打好几次？少来。
			return calculatePrice(price, section);
		}

		@Override
		public BigDecimal handlePrice(final List<ProductPack> participatingPacks, final CampaignSection campaignSection,
				final BigDecimal participatingPrice, final int times, final Map<ProductPack, BigDecimal> finalPriceMap,
				final Map<ProductPack, BigDecimal> preFinalPriceMap) {
			BigDecimal participatedPrice = buildParticipatingPack(participatingPacks, campaignSection, times,
					finalPriceMap, preFinalPriceMap);
			return PriceUtil.keepToCent(participatedPrice);
		}
	},
	/**
	 * 满赠
	 */
	GIFT(3, "满赠", 3),
	/**
	 * 加价购
	 */
	CHEAPEN_OTHER(4, "换购", 4) {
		@Override
		public boolean isForceParticipate() {
			return false;
		}
	},

	PANIC_BUY(6, "抢购", 0){
		@Override
		public boolean isAffectPrice() {
			return true;
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section) {
			final FactorTypeEnum factorType = section.getFactorType();
			if (null != factorType) {
				return factorType.calculate(price, section);
			}
			throw new BusinessLogicNotExpectedException("价格影响因子类型错误");
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section, final int times) {
			return calculatePrice(price, section);
		}

		@Override
		public BigDecimal handlePrice(final List<ProductPack> participatingPacks, final CampaignSection campaignSection,
									  final BigDecimal participatingPrice, final int times, final Map<ProductPack, BigDecimal> finalPriceMap,
									  final Map<ProductPack, BigDecimal> preFinalPriceMap) {
			BigDecimal participatedPrice = buildParticipatingPack(participatingPacks, campaignSection, times,
					finalPriceMap, preFinalPriceMap);
			return PriceUtil.keepToCent(participatedPrice);
		}
	},
	OPTION(7, "N元任选", 2) {
		@Override
		public boolean isAffectPrice() {
			return true;
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section) {
			return section.getFactor();
		}

		@Override
		public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section, final int times) {
			// N元任选应该不支持累加。
			return calculatePrice(price, section);
		}

		@Override
		public BigDecimal handlePrice(final List<ProductPack> participatingPacks, final CampaignSection campaignSection,
				final BigDecimal participatingPrice, final int times, final Map<ProductPack, BigDecimal> finalPriceMap,
				final Map<ProductPack, BigDecimal> preFinalPriceMap) {
			final BigDecimal participatedPrice = calculatePrice(participatingPrice, campaignSection, times);
			// 每一份钱省下的比例。
			final BigDecimal savedRate = participatingPrice.subtract(participatedPrice).divide(participatingPrice, 10,
					RoundingMode.HALF_UP);
			BigDecimal sumPrice= BigDecimal.ZERO;
			for (final ProductPack participatingPack : participatingPacks) {
				BigDecimal rawProductPrice = preFinalPriceMap.get(participatingPack);
				if (rawProductPrice == null) {
					rawProductPrice = participatingPack.getFactProductPrice();
				}
				final BigDecimal participatedProductPrice = PriceUtil
						.keepToCent(rawProductPrice.subtract(rawProductPrice.multiply(savedRate)));
				finalPriceMap.put(participatingPack, participatedProductPrice);
				sumPrice = sumPrice.add(participatedProductPrice);
			}
			// N元任选价格修正
			if(sumPrice.compareTo(participatedPrice)!=0){
				ProductPack participatingPack = participatingPacks.get(0);
				final BigDecimal revisePrize = finalPriceMap.get(participatingPack).add(participatedPrice.subtract(sumPrice));
				finalPriceMap.put(participatingPack,revisePrize);
			}
			return participatedPrice;
		}
	},
	/**
	 * 团购
	 */
	GROUP_BUY(8, "团购", 2);

	public final Integer id;
	public final String name;
	public final Integer sort;

	public static final List<DiscountType> ALL = Collections.unmodifiableList(Arrays.asList(values()));

	DiscountType(final Integer id, final String name, final Integer sort) {
		this.id = id;
		this.name = name;
		this.sort = sort;
	}

	/**
	 * 是否强制参加活动
	 *
	 * @return
	 */
	public boolean isForceParticipate() {
		return true;
	}

	public boolean isAffectPrice() {
		return false;
	}

	/**
	 * 计算价格因子影响后的价格。注：只有第一个满足条件的因子起效。不会对因子列表进行排序。
	 *
	 * @param price
	 * @param section
	 * @return
	 */
	public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section) {
		return price;
	}

	public BigDecimal calculatePrice(final BigDecimal price, final CampaignSection section, final int times) {
		return price;
	}

	@Override
	public Integer getId() {
		return id;
	}

	@Override
	public String getName() {
		return name;
	}

	public Integer getSort() {
		return sort;
	}

	/**
	 * 计算参与活动的商品包中的商品在参与活动之后的价格，放入participatingPacksSavedMap中，然后返回参与后的商品包总价。
	 *
	 * @param participatingPacks
	 * @param campaignSection
	 * @param participatingPrice
	 * @param times
	 * @param finalPriceMap
	 * @param preFinalPriceMap
	 * @return
	 */
	public BigDecimal handlePrice(final List<ProductPack> participatingPacks, final CampaignSection campaignSection,
			final BigDecimal participatingPrice, final int times, final Map<ProductPack, BigDecimal> finalPriceMap,
			final Map<ProductPack, BigDecimal> preFinalPriceMap) {
		for (final ProductPack participatingPack : participatingPacks) {
			finalPriceMap.put(participatingPack, participatingPack.getFactProductPrice());
		}
		return participatingPrice;// 默认是没有影响价格
	}

	protected BigDecimal buildParticipatingPack(List<ProductPack> participatingPacks, CampaignSection campaignSection,
			int times, Map<ProductPack, BigDecimal> finalPriceMap, Map<ProductPack, BigDecimal> preFinalPriceMap) {
		BigDecimal participatedPrice = BigDecimal.ZERO;

		if (null != campaignSection) {
				for (final ProductPack participatingPack : participatingPacks) {
					BigDecimal rawPrice = preFinalPriceMap.get(participatingPack);
					if (rawPrice == null) {
						rawPrice = participatingPack.getFactProductPrice();
					}
					final BigDecimal participatedProductPrice = PriceUtil
							.keepToCent(calculatePrice(rawPrice, campaignSection, times));
					finalPriceMap.put(participatingPack, participatedProductPrice);
					participatedPrice = participatedPrice
							.add(participatedProductPrice.multiply(BigDecimal.valueOf(participatingPack.getCount())));
				}
		} else {
			System.out.println("警告    折扣计算价格  活动为空 ");
		}
		return participatedPrice;
	}

}
