package com.thebeastshop.pegasus.component.campaign;

import java.math.BigDecimal;
import java.math.RoundingMode;

import com.thebeastshop.support.mark.HasIdGetter.HasIntIdGetter;
import com.thebeastshop.support.mark.HasName;

/**
 * 门槛类型，即（订单要满足活动的条件）的类型。
 * 其实就是简单定义了两个维度，是订单能够参与活动的门槛属性。活动指定订单的这两个维度其中之一的值必须满足数量才能够参与活动。
 *
 * @author Liang Wenjian
 */
public enum ConditionType implements HasIntIdGetter, HasName {
	/**
	 * 金额
	 */
	PRICE(1, "金额") {
		@Override
		public boolean matchCondition(final BigDecimal price, final int productCount, final BigDecimal conditionNum) {
			if (conditionNum.compareTo(BigDecimal.ZERO) <= 0) {
				return true;
			}
			return price.compareTo(conditionNum) >= 0;
		}

		@Override
		public int cumulate(final BigDecimal line, final BigDecimal participatePrice, final int participateCount) {
			return participatePrice.divide(line, RoundingMode.HALF_UP).intValue();
		}
	},
	/**
	 * 商品数量
	 */
	PRODUCT_COUNT(2, "商品数量") {
		@Override
		public boolean matchCondition(final BigDecimal price, final int productCount, final BigDecimal conditionNum) {
			if (conditionNum.compareTo(BigDecimal.ZERO) <= 0) {
				return true;
			}
			return productCount >= conditionNum.intValue();
		}

		@Override
		public int cumulate(final BigDecimal line, final BigDecimal participatePrice, final int participateCount) {
			return (int) (participateCount / line.intValue());
		}
	};

	public final Integer id;
	public final String name;

	ConditionType(final Integer id, final String name) {
		this.id = id;
		this.name = name;
	}

	public abstract boolean matchCondition(final BigDecimal price, final int productCount,
			final BigDecimal conditionNum);

	@Override
	public Integer getId() {
		return id;
	}

	@Override
	public String getName() {
		return name;
	}

	public abstract int cumulate(BigDecimal line, BigDecimal participatePrice, int participateCount);
}
