package com.thebeastshop.pegasus.component.support.service.impl;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.thebeastshop.pegasus.component.support.service.RedisService;
import com.thebeastshop.support.util.JsonUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author gongjun
 * @since 2016-07-25
 */
@Component
public class RedisServiceImpl implements RedisService {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private JedisPool pool;


    public void expire(final String key, final Integer seconds) {
        Jedis jedis = null;
        try {
            jedis = pool.getResource();
            jedis.expire(key, seconds);
        } catch (Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    public String get(final String key) {
        String str = "";
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                str = jedis.get(key);
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }

        return str;
    }

    public <T> T get(final String key, final Type clazz) {
        T t = null;
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                String value = jedis.get(key);
                if (StringUtils.isEmpty(value)) {
                    return null;
                }
//                t = JsonUtil.toObject(value, clazz);
                ObjectMapper mapper = new ObjectMapper();
                mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                mapper.configure(DeserializationFeature.FAIL_ON_NULL_FOR_PRIMITIVES, false);

                t = mapper.readValue(value, mapper.getTypeFactory().constructType(clazz));
//                t = JSON.parseObject(value, clazz);
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }

        return t;
    }

    public void set(final String key, final String value) {
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(key)) {
                jedis = this.pool.getResource();
                jedis.set(key, value);
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    public <T> void set(final String key, final T value) {
        try {
            if (value == null) {
                this.set(key, "");
                return;
            }
            this.set(key, JsonUtil.toJson(value));
        } catch (final Exception e) {
            logger.error(e.getMessage());
        }
    }

    public void set(final String key, final String value, final int seconds) {
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                jedis.set(key, value);
                jedis.expire(key, seconds);
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    public <T> void set(final String key, final T value, final int seconds) {
        try {
            this.set(key, JsonUtil.toJson(value), seconds);
        } catch (final Exception e) {
            logger.error(e.getMessage());
        }
    }

    public boolean exists(final String key) {
        boolean flag = false;
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                flag = jedis.exists(key);
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return flag;
    }

    public void del(final String key) {
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                jedis.del(key);
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }
    @Override
    public Long incr(String key) {
        Jedis jedis = null;
        try {
            if(StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                return jedis.incr(key);
            } else {
                logger.error("redis key cannot be null or blank");
            }
        } catch (final Exception e ){
            logger.error(e.getMessage());
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
        return -1L;
    }

    @Override
    public void delsBykeyPattern(String keyPattern){
        Jedis jedis = null;
        try {
            if (StringUtils.isNotBlank(keyPattern)) {
                jedis = pool.getResource();
                Set<String> keys =  jedis.keys(keyPattern);
                if(CollectionUtils.isNotEmpty(keys)){
                    String[] delKeys = (String[]) keys.toArray(new String[keys.size( )]);
                    jedis.del(delKeys);
                }
            } else {
                logger.error("redis key cannot be blank");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }
    @Override
    public Long decr(String key) {
        Jedis js = null;
        try{
            if(StringUtils.isNotBlank(key)) {
                js = pool.getResource();
                return js.decr(key);
            } else {
                logger.error("Redis key cannot be null or empty");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if(null != js) {
                js.close();
            }
        }
        return -1L;
    }

    @Override
    public Long lpush(String key, String... values) {
        Jedis jedis = null;
        try {
            if(StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                return jedis.lpush(key, values);
            } else {
                logger.error("Redis key cannot be null or empty");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
        return -1L;
    }

    @Override
    public List<String> lrange(String key) {
        Jedis jedis = null;
        try {
            if(StringUtils.isNotBlank(key)) {
                jedis = pool.getResource();
                return jedis.lrange(key, 0, -1);
            } else {
                logger.error("Redis key cannot be null or empty");
            }
        } catch (final Exception e) {
            logger.error(e.getMessage());
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
        return null;
    }

    @Override
    public void hmset(String key, Map<String, String> hash, int second) {
        Jedis jedis = null;
        try {
            jedis = pool.getResource();
            Pipeline pipelined = jedis.pipelined();
            pipelined.hmset(key, hash);
            pipelined.expire(key, second);
            pipelined.sync();
        } catch (Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    @Override
    public List<String> hmget(String key, String... fields) {
        Jedis jedis = null;
        try {
            jedis = pool.getResource();
            List<String> resultList = jedis.hmget(key, fields);
            return resultList;
        } catch (Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return null;
    }
}
