package com.thebeastshop.pegasus.component.file.service.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;
import com.thebeastshop.pegasus.component.file.config.Config;
import com.thebeastshop.pegasus.component.file.domain.FileInfo;
import com.thebeastshop.pegasus.component.file.service.FileService;
import com.thebeastshop.support.exception.UnknownException;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.file.FileConstant;
import com.thebeastshop.support.util.ByteUtil;

@Service
public class FileServiceImpl implements FileService {

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	private static OSSClient ossClient;

	// ================pivate===============
	/**
	 * 获取ossClient
	 * 
	 * @return
	 */
	private OSSClient getOssClient() {
		if (ossClient == null) {
			ossClient = new OSSClient(Config.OSS_END_POINT, Config.OSS_ACCESS_ID, Config.OSS_ACCESS_KEY);
		}
		return ossClient;
	}
	// =====================================

	@Override
	public FileInfo uploadFile(String localFilePath, String extraPath) {
		File file = new File(localFilePath);
		return this.uploadFile(file, extraPath);
	}

	@Override
	public FileInfo uploadFile(File file, String extraPath) {
		InputStream input = null;
		try {
			input = new FileInputStream(file);
		} catch (Exception e) {
			logger.error("上传文件失败: {}", e);
			throw new UnknownException("上传文件失败: " + e.getMessage());
		}

		return this.uploadFile(input, extraPath);
	}

	@Override
	public FileInfo uploadFile(InputStream inputStream, String extraPath) {
		try {
			// 数据流提取出字节数组
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			byte[] buffer = new byte[1024];
			int len;
			while ((len = inputStream.read(buffer)) > -1) {
				baos.write(buffer, 0, len);
			}
			inputStream = new ByteArrayInputStream(baos.toByteArray());

			// check文件类型
			String key = "";
			byte[] b = Arrays.copyOf(baos.toByteArray(), 28);
			String hex = ByteUtil.bytes2Hex(b).toUpperCase();
			Set<String> keys = FileConstant.FILE_TYPE.keySet();
			for (String k : keys) {
				if (hex.startsWith(k)) {
					key = k;
				}
			}
			if (StringUtils.isBlank(key)) {
				throw new WrongArgException("文件类型不支持", "文件魔数值", hex);
			}

			String fileType = FileConstant.FILE_TYPE.get(key);
			String fileName = FileConstant.BASE_PATH + this.buildExtraPath(extraPath)
					+ UUID.randomUUID().toString().replace("-", "") + "." + fileType;
			getOssClient().putObject(Config.OSS_BUCKET_NAME, fileName, inputStream);

			FileInfo info = new FileInfo(inputStream, FileConstant.BASE_URL + fileName);
			return info;
		} catch (WrongArgException e) {
			logger.error("上传文件异常: {}", e);
			throw e;
		} catch (Exception e) {
			logger.error("上传文件异常: {}", e);
			throw new UnknownException("上传文件失败: " + e.getMessage());
		}
	}

	@Override
	public List<FileInfo> listFiles(String extraPath) {
		try {
			ObjectListing listing = getOssClient().listObjects(Config.OSS_BUCKET_NAME,
					FileConstant.BASE_PATH + this.buildExtraPath(extraPath));
			List<FileInfo> list = new LinkedList<FileInfo>();
			for (OSSObjectSummary objectSummary : listing.getObjectSummaries()) {
				list.add(this.getFile(FileConstant.BASE_URL + objectSummary.getKey()));
			}
			return list;
		} catch (Exception e) {
			logger.error("获取文件列表异常: {}", e);
			throw new UnknownException("获取文件列表失败: " + e.getMessage());
		}
	}

	@Override
	public FileInfo getFile(String fileUrl) {
		try {
			OSSObject ossObject = getOssClient().getObject(Config.OSS_BUCKET_NAME, this.getKey(fileUrl));
			FileInfo info = new FileInfo(ossObject.getObjectContent(), fileUrl);
			return info;
		} catch (Exception e) {
			logger.error("获取文件异常: {}", e);
			throw new UnknownException("获取文件失败: " + e.getMessage());
		}
	}

	@Override
	public void deleteFile(String fileUrl) {
		try {
			String key = this.getKey(fileUrl);
			getOssClient().deleteObject(Config.OSS_BUCKET_NAME, key);
		} catch (Exception e) {
			logger.error("删除文件异常: {}", e);
			throw new UnknownException("删除文件失败: " + e.getMessage());
		}
	}

	private String getKey(String fileUrl) {
		return fileUrl.replace(FileConstant.BASE_URL, "");
	}

	/**
	 * 处理额外路径
	 * 
	 * @param extraPath
	 * @return
	 */
	private String buildExtraPath(String extraPath) {
		if (StringUtils.isNotBlank(extraPath)) {
			extraPath = extraPath.indexOf("/") == 0 ? extraPath.substring(1) : extraPath;
			extraPath += extraPath.lastIndexOf("/") == -1 ? "/" : "";
		} else {
			extraPath = "";
		}
		return extraPath;
	}

	public static void main(String[] args) {
		FileService service = new FileServiceImpl();
		service.listFiles("");
	}

}
