package com.thebeastshop.pegasus.component.coupon.service.impl;

import java.util.Collection;
import java.util.List;

import com.thebeastshop.pegasus.component.coupon.domain.CouponCount;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.component.coupon.condition.CouponCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponDao;
import com.thebeastshop.pegasus.component.coupon.domain.Coupon;
import com.thebeastshop.pegasus.component.coupon.domain.CouponSample;
import com.thebeastshop.pegasus.component.coupon.domain.impl.CouponWrapper;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponImpl;
import com.thebeastshop.pegasus.component.coupon.domain.impl.DefaultCouponSampleImpl;
import com.thebeastshop.pegasus.component.coupon.service.BaseService;
import com.thebeastshop.pegasus.component.coupon.service.CouponSampleService;
import com.thebeastshop.pegasus.component.coupon.service.CouponService;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.exception.WrongStateException;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class CouponServiceImpl extends BaseService implements CouponService {

	@Autowired
	private CouponSampleService couponSampleService;

	@Autowired
	private CouponDao couponDao;

	@Override
	public Coupon getById(long id) {
		Coupon obj = this.couponDao.getById(id);
		if (obj == null) {
			throw new WrongArgException("优惠券模板不存在", "id", id);
		}
		return obj;
	}

	@Override
	public List<Coupon> getCoupons(long ownerId) {
		return this.couponDao.getCoupons(ownerId);
	}

	@Override
	public List<Coupon> getCoupons(long ownerId, Long couponSampleId) {
		return this.couponDao.getCoupons(ownerId, couponSampleId);
	}

	@Override
	public List<Coupon> getUsableCoupons(final long ownerId) {
		return this.couponDao.getUsableCoupons(ownerId);
	}

	@Override
	public List<Coupon> getExpiredCoupons(final long ownerId) {
		return this.couponDao.getExpiredCoupons(ownerId);
	}

	@Override
	public List<Coupon> getUsedCoupons(final long ownerId) {
		return this.couponDao.getUsedCoupons(ownerId);
	}

	@Override
	public List<Coupon> getCouponsByCondition(CouponCondition condition) {
		return this.couponDao.getCouponsByCondition(condition);
	}

	@Override
	public CouponCount queryCouponCount(CouponCondition condition) {
		return this.couponDao.countCouponByCondition(condition);
	}

	@Override
	public Coupon create(final Coupon coupon) {
		return this.couponDao.create(coupon);
	}

	@Override
	public void invalid(final Coupon coupon) {
		Coupon couponImpl = new CouponWrapper<Coupon>(coupon) {
			@Override
			public boolean isValid() {
				return false;
			}
		};
		this.update(couponImpl);
	}

	@Transactional
	@Override
	public Coupon send(CouponSample couponSample, long memberId) {
		if (couponSample == null) {
			throw new WrongArgException("发送优惠券失败", "couponSample", couponSample);
		}
		// 检查剩余数量
		if (couponSample.getBalanceCount() <= 0) {
			throw new WrongArgException("优惠券已发完");
		}
		// 检查每人最多获得的上限限制
		Integer perMemberCount = this.getCouponsCount(memberId, couponSample.getId());
		if (perMemberCount >= couponSample.getMaxPerMember()) {
			throw new WrongStateException("获得" + couponSample.getName() + "的优惠券已达到上限。",
					"member{id=" + memberId + "}.coupons{couponSampleId=" + couponSample.getId() + "}.size",
					couponSample.getMaxPerMember());
		}
		// 优惠券剩余数量减一
		DefaultCouponSampleImpl couponSampleImpl = (DefaultCouponSampleImpl) couponSample;
		couponSampleImpl.setBalanceCount(couponSampleImpl.getBalanceCount() - 1);
		this.couponSampleService.update(couponSampleImpl);
		// 创建优惠券
		DefaultCouponImpl coupon = new DefaultCouponImpl();
		coupon.setCouponSampleId(couponSample.getId());
		coupon.setExpireTime(couponSample.getExpireTime());
		coupon.setNote(couponSample.getNote());
		coupon.setOwnerId(memberId);
		coupon.setStartTime(couponSample.getStartTime());
		coupon.setUsed(false);
		coupon.setValid(true);
		return this.create(coupon);
	}

	@Override
	public Boolean update(Coupon obj) {
		return this.couponDao.update(obj);
	}

	@Override
	public void use(Coupon obj) {
		if (obj != null) {
			Coupon coupon = new CouponWrapper<Coupon>(obj) {
				@Override
				public boolean isUsed() {
					return true;
				}
			};
			this.update(coupon);
		}
	}

	@Override
	public void use(Long couponId) {
		use(getById(couponId));
	}

	@Override
	public int getCouponsCount(long ownerId) {
		return this.getCoupons(ownerId).size();
	}

	@Override
	public int getCouponsCount(long ownerId, Long couponSampleId) {
		return this.getCoupons(ownerId, couponSampleId).size();
	}

	@Override
	public int getUsableCouponsCount(long ownerId) {
		return getUsableCoupons(ownerId).size();
	}

	@Override
	public int getExpiredCouponsCount(long ownerId) {
		return getExpiredCoupons(ownerId).size();
	}

	@Override
	public int getUsedCouponsCount(long ownerId) {
		return getUsedCoupons(ownerId).size();
	}

	@Override
	public Collection<Coupon> create(Collection<Coupon> coupons) {
		List<Coupon> list = Lists.newArrayList();
		for (Coupon coupon : coupons) {
			list.add(create(coupon));
		}
		return list;
	}

	@Override
	public void invalid(Collection<Coupon> coupons) {
		for (Coupon coupon : coupons) {
			invalid(coupon);
		}

	}

	@Override
	public Collection<Coupon> send(CouponSample couponSample, Collection<Long> memberIds) {
		List<Coupon> list = Lists.newArrayList();
		for (Long memberId : memberIds) {
			list.add(send(couponSample, memberId));
		}
		return list;
	}

}
