package com.thebeastshop.trans.enums;

import com.google.common.collect.Lists;
import com.thebeastshop.trans.exception.TsWrongStateException;

import java.util.Collections;
import java.util.List;

/**
 * 订单流程状态(后端订单状态)。
 *
 * @author Liang Wenjian
 */
public enum TsOrderFlowStatusEnum {

	CANCELLED(0, "已取消") {
		@Override
		public TsOrderFlowStatusEnum cancel() {
			return this;
		}
	},
	CONFIRMED(1, "未付款") {
		@Override
		public TsOrderFlowStatusEnum confirm() {
			return this;
		}

		@Override
		public TsOrderFlowStatusEnum pay() {
			return PAID;
		}

		@Override
		public TsOrderFlowStatusEnum payTimeout() {
			return CANCELLED;
		}

		@Override
		public TsOrderFlowStatusEnum customerHangUp() {
			return CUSTOMER_HUNG_UP;
		}

		@Override
		public TsOrderFlowStatusEnum cancel() {
			return CANCELLED;
		}
	},
	CUSTOMER_HUNG_UP(2, "未付款挂起"),
	/**
	 * 后台创建订单时的状态
	 */
	CREATED(3, "待审批") {
		@Override
		public TsOrderFlowStatusEnum confirm() {
			return CONFIRMED;
		}

		@Override
		public TsOrderFlowStatusEnum cancel() {
			return CANCELLED;
		}
	},
	CHANGE_AFTER_RECEIVE(5, "换货待收货完成"), AUDIT_HUNG_UP(9, "审单挂起"), PAID(10,
			"待审单"), AUDITED(11, "已审单"), DONE(12, "已完成"), ;

	/**
	 * 已确认的订单状态组
	 */
	public static final List<TsOrderFlowStatusEnum> IS_CONFIRMED_ARRAYS = Collections
			.unmodifiableList(Lists.newArrayList(CONFIRMED, CUSTOMER_HUNG_UP));

	public static boolean isConfirmed(int id) {
		for (TsOrderFlowStatusEnum state : IS_CONFIRMED_ARRAYS) {
			if (state.getId().intValue() == id)
				return true;
		}
		return false;
	}

	/**
	 * 已支付的订单状态组
	 */
	public static final List<TsOrderFlowStatusEnum> IS_PAID_ARRAYS = Collections
			.unmodifiableList(Lists.newArrayList(AUDIT_HUNG_UP,PAID, AUDITED, DONE));

	public static boolean isPaid(int id) {
		for (TsOrderFlowStatusEnum state : IS_PAID_ARRAYS) {
			if (state.getId().intValue() == id)
				return true;
		}
		return false;
	}

	/**
	 * 已取消的订单状态组
	 */
	public static final List<TsOrderFlowStatusEnum> IS_CANCELLED_ARRAYS = Collections
			.unmodifiableList(Lists.newArrayList(CANCELLED));

	public static boolean isCanclled(int id) {
		for (TsOrderFlowStatusEnum state : IS_CANCELLED_ARRAYS) {
			if (state.getId().intValue() == id)
				return true;
		}
		return false;
	}

	public final int id;
	public final String name;

	TsOrderFlowStatusEnum(final int id, final String name) {
		this.id = id;
		this.name = name;
	}

	public Integer getId() {
		return id;
	}

	public String getName() {
		return name;
	}

	public TsOrderFlowStatusEnum confirm() {
		throw new TsWrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 取消
	 *
	 * @return
	 */
	public TsOrderFlowStatusEnum cancel() {
		throw new TsWrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 超时未支付
	 *
	 * @return
	 */
	public TsOrderFlowStatusEnum payTimeout() {
		throw new TsWrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 客户挂起
	 *
	 * @return
	 */
	public TsOrderFlowStatusEnum customerHangUp() {
		throw new TsWrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 支付
	 *
	 * @return
	 */
	public TsOrderFlowStatusEnum pay() {
		throw new TsWrongStateException("订单状态不对。", "state", this);
	}


	public static TsOrderFlowStatusEnum getEnumById(Integer id) {
		for (TsOrderFlowStatusEnum e : TsOrderFlowStatusEnum.values()) {
			if (e.getId().equals(id)) {
				return e;
			}
		}
		return null;
	}

}
