package com.thebeastshop.stock.service;

import com.thebeastshop.common.*;
import com.thebeastshop.exchange.enums.ExchgTypeEnum;
import com.thebeastshop.litx.annotations.Compensable;
import com.thebeastshop.stock.dto.*;
import com.thebeastshop.stock.vo.*;

import java.util.List;
import java.util.Map;

public interface SStockService {


	/**
	 * 获取SKU在某个仓库下的库存
	 * @param skuCode
	 * @return 库存
	 */
	SSkuStockVO getSkuStock(String skuCode, String warehouseCode);


	/**
	 * 批量获取SKU在某个仓库下的库存
	 * @param skuCodes
	 * @return Map<SKU编号, 库存>
	 */
	Map<String, SSkuStockVO> getSkuStocks(List<String> skuCodes, String warehouseCode);


	/**
	 * 获取SKU在一组仓库列表下的库存
	 * @param skuCode
	 * @param warehouseCodes
	 * @return Map<逻辑仓编号, 库存>
	 */
	Map<String, SSkuStockVO> getSkuStocks(String skuCode, List<String> warehouseCodes);


	/**
	 * 根据条件查询SKU库存
	 * @param skuStockQueryDTOList
	 * @return
     */
	List<SSkuStockVO> getSkuStocks(List<SSkuStockQueryDTO> skuStockQueryDTOList);


	/**
	 * 批量获取SKU列表在一组仓库列表下的库存
	 * @param skuCodes
	 * @param warehouseCodes
	 * @return Map<逻辑仓编号, Map<SKU编号, 库存>>
     */
	Map<String, Map<String, SSkuStockVO>> getSkuStocks(List<String> skuCodes, List<String> warehouseCodes);


	/**
	 * 获取SKU在所有仓库下的库存
	 * @param skuCode
	 * @return Map<逻辑仓编号, 库存>
     */
	Map<String, SSkuStockVO> getSkuStocksInAllWarehouses(String skuCode);


	/**
	 * 批量查询SKU在所有仓库下的库存
	 * @param skuCodes
	 * @return Map<Sku CODE -> MAP<Warehouse CODE -> STOCK>>
	 */
	Map<String, Map<String, SSkuStockVO>> getSkuStocksInAllWarehouses(List<String> skuCodes);


	/**
	 * 分页查询仓库中的所有SKU编号
	 * @return
     */
	PageQueryResp<String> getSkuListInAllWarehouse(PageCond page);


	/**
	 * 分页查询仓库中所有SKU的可用库存
	 * @param page
	 * @return PAGE<MAP<Warehouse CODE -> STOCK>>
     */
	PageQueryResp<Map<String, SSkuStockVO>> getAllSkuStocksInAllWarehouse(PageCond page);

	/**
	 * 分页查询在某个仓库中所有SKU的可用库存
	 * @param page
	 * @return PAGE<MAP<Warehouse CODE -> STOCK>>
	 */
	PageQueryResp<SSkuStockVO> getAllSkuStocksInWarehouse(String warehouseCode, PageCond page);
	List<SSkuStockVO> getAllSkuStocksInWarehouse(String warehouseCode, List<String> skuCodeList);
	List<String> findSkuListInWarehouse(String warehouseCode, PageCond page);

	/**
	 * 记录库存
	 * @param record
	 * @return
     */
	@Compensable
	ServiceResp<SStockRecordVO> record(SStockRecordVO record);

	@Compensable
    ServiceResp<List<SStockRecordVO>> batchRecord(List<SStockRecordVO> recordList);

	/**
	 * 预备库存
	 * @param prepareDTO
	 * @return
     */
	@Compensable
	ServiceResp<SOccupyResultVO> prepare(SStockPrepareDTO prepareDTO);

	/**
	 * 批量预备库存
	 * @return
     */
	ServiceResp<List<SOccupyResultVO>> prepare(List<SStockPrepareDTO> prepareDTOList);

	ServiceResp<SOccupyResultVO> prepare();

	/**
	 * 增加预备库存数量
	 * @param quantityChangeDTO
	 * @return
	 */
	@Compensable
	ServiceResp<SStockPrepareQuantityChangeDTO> incrementPreparedQuantity(SStockPrepareQuantityChangeDTO quantityChangeDTO);

	/**
	 * 减少预备库存数量
	 * @param quantityChangeDTO
	 * @return
	 */
	@Compensable
	ServiceResp<SStockPrepareQuantityChangeDTO> decrementPreparedQuantity(SStockPrepareQuantityChangeDTO quantityChangeDTO);

	/**
	 * 占用库存
	 * @param occupyDTO
	 * @return
     */
	@Compensable
	ServiceResp<SOccupyResultVO> occupy(SStockOccupyDTO occupyDTO);

	/**
	 * 批量占用库存
	 * @param occupyDTOList
	 * @return
     */
	@Compensable
	ServiceResp<List<SOccupyResultVO>> occupy(List<SStockOccupyDTO> occupyDTOList);

	/**
	 * 释放后占用
	 * @param occupyDTOList
	 * @param releaseOccupationDTOList
     * @return
     */
	@Compensable
	ServiceResp<SOccupyAfterReleaseResultVO> occupyAfterRelease(
			List<SStockOccupyDTO> occupyDTOList,
			List<SStockReleaseDTO> releaseOccupationDTOList);

	/**
	 * 释放后占用 (不影响缓存)
	 * @param occupyDTOList
	 * @param releaseOccupationDTOList
	 * @return
	 */
	@Compensable
	ServiceResp<SOccupyAfterReleaseResultVO> occupyAfterReleaseWithoutCache(
			List<SStockOccupyDTO> occupyDTOList,
			List<SStockReleaseDTO> releaseOccupationDTOList);

	/**
	 * 释放占用库存
	 * @param releaseOccupationDTO
	 * @return
     */
	@Compensable
	ServiceResp<SReleaseResultVO> release(SStockReleaseDTO releaseOccupationDTO);

	/**
	 * 批量释放占用库存
	 * @param releaseOccupationDTOList
	 * @return
     */
	@Compensable
	ServiceResp<List<SReleaseResultVO>> release(List<SStockReleaseDTO> releaseOccupationDTOList);

	/**
	 * 根据Id占用库存
	 * @param id
	 * @return
     */
	ServiceResp<SReleaseResultVO> releaseById(Long id);

	/**
	 * 根据Id列表批量占用库存
	 * @param ids
	 * @return
     */
	ServiceResp<List<SReleaseResultVO>> releaseByIds(List<Long> ids);

	/**
	 * 获取SKU在所有仓库下的实际库存
	 * @param skuCode
	 * @return Map<仓库编号, SSkuStockQuantityVO>
     */
	Map<String, SSkuStockQuantityVO> getSkuRealStocksInAllWarehouses(String skuCode);

	/**
	 * 获取SKU在某个仓库下的实际库存
	 * @param skuCode
	 * @param warehouseCode
     * @return
     */
	SSkuStockQuantityVO getSkuRealStock(String skuCode, String warehouseCode);


	/**
	 * 批量获取SKU在某个仓库下的实际库存
	 * @param skuCodes
	 * @param warehouseCode
     * @return Map<SKU编号, SSkuStockQuantityVO>
     */
	Map<String, SSkuStockQuantityVO> getSkuRealStocks(
			List<String> skuCodes, String warehouseCode);


	/**
	 * 根据SKU编号获取在所有仓库下的占用库存数
	 * @param skuCode
	 * @return Map<仓库编号, SSkuStockQuantityVO>
     */
	Map<String, SSkuStockQuantityVO> getSkuOccupyStocksInAllWarehouses(String skuCode);

	/**
	 * 批量获取SKU在某个仓库下的占用库存数
	 * @param skuCodes
	 * @param warehouseCode
     * @return Map<SKU编号, SSkuStockQuantityVO>
     */
	Map<String, SSkuStockQuantityVO> getSkuOccupyStocks(
			List<String> skuCodes, String warehouseCode);

	/**
	 * 根据条件获取库存记录信息
	 * @param cond
	 * @return
     */
	List<SStockRecordVO> findStockRecordByCond(SStockRecordConditionDTO cond);

	Integer findStockRecordCountByCond(SStockRecordConditionDTO cond);

	/**
	 * 查询 SPU 库存明细
	 *
	 * @param queryList {@link SSpuStockDetailQueryDTO} 对象列表
	 * @return Map&lt 商品编号, {@link SSpuStockDetailVO} 对象实例 &gt
	 */
	Map<String, SSpuStockDetailVO> findSpuStockDetail(List<SSpuStockDetailQueryDTO> queryList);

	/**
	 * 查询 SPV 库存明细
	 * <p>
	 * {@link SSpuStockDetailQueryDTO} 参数类型为接口类，其有以下几种实现类:
	 * <ul>
	 *     <li>普通 SPV 库存 {@link SSpuStockDetailQueryDTO}: <P> {@code SStocks.spvDetail(Long)} 方法构造</li>
	 *     <li>组合商品 SPV 库存 {@link SSpvCombinedStockDetailQueryDTO}: <P> {@code SStocks.spvCombinedDetail(Long)} 方法构造</li>
	 * </ul>
	 *
	 * @param queryList {@link SAbstractSpvStockDetailQueryDTO} 对象列表
	 * @return Map&lt SPV ID, {@link SSpvStockDetailVO} 对象实例 &gt
	 * @see com.thebeastshop.stock.SStocks#spvDetail(Long)
	 * @see com.thebeastshop.stock.SStocks#spvCombinedDetail(Long)
	 */
	Map<Long, SSpvStockDetailVO> findSpvStockDetail(List<SSpvStockDetailQuery> queryList);

	/**
	 * 查询 iPos 的 SPU 库存明细
	 * <p>
	 * {@link SIPosSpuStockDetailQueryDTO}类型参数对象通过 {@code SStock.iPosSpuDetail(String)} 方法来构造
	 *
	 * @param queryList {@link SIPosSpuStockDetailQueryDTO} 对象列表
	 * @return Map&lt 商品编号, {@link SIPosSpuStockDetailVO} 对象实例 &gt
	 * @see com.thebeastshop.stock.SStocks#iPosSpuDetail(String)
	 */
	Map<String, SIPosSpuStockDetailVO> findIPosSpuStockDetail(List<SIPosSpuStockDetailQueryDTO> queryList);

	/**
	 * 查询 iPos 的 SPV 库存明细
	 * <p>
	 * {@link SIPosSpvStockDetailQuery} 参数类型为接口类，其有以下几种实现类:
	 * <ul>
	 *     <li>普通 iPos SPV 库存 {@link SIPosSpvStockDetailQueryDTO}: <P> {@code SStocks.iPosSpvDetail(Long)} 方法构造</li>
	 *     <li>组合 iPos SPV 库存 {@link SIPosSpvCombinedStockDetailQueryDTO}: <P> {@code SStocks.iPosSpvCombinedDetail(Long)} 方法构造</li>
	 * </ul>
	 *
	 * @param queryList {@link SIPosSpvStockDetailQueryDTO} 对象列表
	 * @return {@link SIPosSpvStockDetailVO} 对象列表
	 * @see com.thebeastshop.stock.SStocks#iPosSpvDetail(Long)
	 * @see com.thebeastshop.stock.SStocks#iPosSpvCombinedDetail(Long)
	 */
	Map<Long, SIPosSpvStockDetailVO> findIPosSpvStockDetail(List<SIPosSpvStockDetailQuery> queryList);


	/**
	 * 根据条件获取占用记录信息
	 *
	 * @param cond
	 * @return
     */
	List<SStockOccupyRecordVO> findStockRecordInvOccupyByCond(SStockOccupyDTO cond);


	List<SStockOccupyRecordVO> findStockRecordInvOccupy(List<String> skuCodes, List<String> warehouseCodes);

	/**
	 * 根据条件获取待入库总数信息
	 * @param cond
	 * @return
     */
	SStockOccupyRecordVO findCountStockRecordInvRcdWillInByCond(SStockOccupyDTO cond);

	/**
	 * 根据条件获取待入库记录信息
	 * @param cond
	 * @return
     */
	List<SStockOccupyRecordVO> findStockRecordInvRcdWillInByCond(SStockOccupyDTO cond);

	/**
	 * 根据referenceCode获取库存占用记录
	 * @param referenceCode
	 * @return
     */
	SStockOccupyRecordVO getOccupyRecordsByReferenceCode(String referenceCode);

	/**
	 * 根据referenceCode批量获取库存占用记录
	 * @param referenceCodes
	 * @return
     */
	List<SStockOccupyRecordVO> getOccupyRecordsByReferenceCodes(List<String> referenceCodes);


	/**
	 * 根据referenceCode批量获取库存占用记录(like)
	 * @param referenceCode
	 * @return
     */
	List<SStockOccupyRecordVO> getOccupyRecordsByReferenceCodeLike(String referenceCode);




	/**
	 * 更新占用类型
	 * @param referenceCode
	 * @param occupyType
     * @return
     */
	ServiceResp<Boolean> updateOccupyType(String referenceCode, Integer occupyType);

	/**
	 * 更新引用编号
	 * @param oldReferenceCode
	 * @param newReferenceCode
     * @return
     */
	ServiceResp<Boolean> updateReferenceCode(String oldReferenceCode, String newReferenceCode);

	/**
	 * 逻辑仓是否存在库存
	 * @param warehouseCode
	 * @return
	 */
	ServiceResp<Boolean> warehouseHasStock(String warehouseCode);

	/**
	 * 删除占用记录
	 * @param deleteOccupationDTOList
	 * @return
     */
//	ServiceResp<Boolean> deleteOccupation(List<SStockDeleteOccupationDTO> deleteOccupationDTOList);

	/**
	 * 获取预备库存数
	 * @param preparedStockQueryDTO
	 * @return
     */
	SPreparedStockVO getPreparedQuantity(SPreparedStockQueryDTO preparedStockQueryDTO);

	/**
	 * 批量获取预备库存数
	 * @param preparedStockQueryList
	 * @return
	 */
	List<SPreparedStockVO> getPreparedQuantityList(List<SPreparedStockQueryDTO> preparedStockQueryList);





	//////////////////////////////////////////////////////////////////////////////////////////////////
	// 回滚接口
	//////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * 回滚 - 记录库存
	 * @param resp
	 * @return
	 */
	ServiceResp<Boolean> rollbackRecord(ServiceResp<SStockRecordVO> resp);


	ServiceResp<Boolean> rollbackBatchRecord(ServiceResp<List<SStockRecordVO>> recordList);

	/**
	 * 回滚 - 记录库存
	 * @param recordList
	 * @return
	 */
	ServiceResp<Boolean> rollbackRecord(List<ServiceResp<SStockRecordVO>> recordList);

	/**
	 * 回滚 - 占用库存
	 * @param resp
	 * @return
     */
	ServiceResp<Boolean> rollbackOccupy(ServiceResp resp);


	/**
	 * 回滚 - 释放占用库存
	 * @param resp
	 * @return
     */
	ServiceResp<Boolean> rollbackRelease(ServiceResp resp);


	/**
	 * 回滚 - 预备库存
	 * @param resp
	 * @return
     */
	ServiceResp<Boolean> rollbackPrepare(ServiceResp resp);


	/**
	 * 回滚 - 增加预备库存
	 * @param resp
	 * @return
	 */
	ServiceResp<Boolean> rollbackIncrementPreparedQuantity(ServiceResp resp);

	/**
	 * 回滚 - 减少预备库存
	 * @param resp
	 * @return
	 */
	ServiceResp<Boolean> rollbackDecrementPreparedQuantity(ServiceResp resp);


	/**
	 * 回滚 - 释放后占用
	 * @param resp
	 * @return
     */
	ServiceResp<Boolean> rollbackOccupyAfterRelease(ServiceResp resp);

	/**
	 * 回滚 - 释放后占用
	 * @param resp
	 * @return
	 */
	ServiceResp<Boolean> rollbackOccupyAfterReleaseWithoutCache(ServiceResp resp);

	/**
	 * 查询积分兑换 库存
	 * @param pointExchangeQueryDTO
	 * @return
	 */
  ServiceResp<SSkuPointExchangeStockVO> getPointExchangeStock(SSkuPointExchangeQueryDTO pointExchangeQueryDTO);

	/**
	 * 批量查询积分兑换 库存
	 * @param pointExchangeQueryDTOList
	 * @return
	 */
  ServiceResp<List<SSkuPointExchangeStockVO>> getPointExchangeStocks(List<SSkuPointExchangeQueryDTO> pointExchangeQueryDTOList);

	/**
	 * 批量查询 单个兑换类型 库存
	 * @param exchgType
	 * @param referenceCodeList
	 * @return
	 */
  ServiceResp<List<SSkuPointExchangeStockVO>> getPointExchangeStocks(ExchgTypeEnum exchgType, List<String> referenceCodeList);

	/**
	 * 查询库存有效期内的SKU（部分SKU长时间没有出入库记录，定义为无效SKU）
	 * @param warehouseCode
	 * @param skuCodes
	 * @param days
	 * @return
	 */
  List<String> getWithinPeriodSku(String warehouseCode, List<String> skuCodes, Integer days);
}
