package com.thebeastshop.stock.service;

import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.stock.dto.SLockSpotStockDTO;
import com.thebeastshop.stock.dto.SLockedPromptStockQueryDTO;
import com.thebeastshop.stock.dto.SPresaleVirtualQuantityChangeDTO;
import com.thebeastshop.stock.dto.SStockPrepareQuantityChangeDTO;
import com.thebeastshop.stock.vo.SLockSpotStockResultVO;
import com.thebeastshop.stock.vo.SVirtualStockVO;
import com.thebeastshop.stock.vo.SReleaseSpotStockResultVO;

/**
 * 预售虚拟库存服务
 * <p>处理包括权益预售库存在内的预售虚拟库存操作</p>
 *
 * @author gongjun
 * @author yuancheng.zhang
 */
public interface SPresaleVirtualStockService {

    /**
     * 创建预售虚拟库存，在不同业务下有不同解释（如：创建权益预售库存）
     * <p>影响的表：</p>
     * <pre>
     * t_op_presale_virtual_stock
     * </pre>
     * <p>业务场景（创建权益预售）：</p>
     * <pre>
     * 在创建预售的时候调用该接口，记录下权益总数和权益预售已分配数，
     * 并在随后的相关业务操作中更新这些数据
     * </pre>
     *
     * @param stockVO 预售虚拟库存，{@link SVirtualStockVO}实例
     * @return 插入完的预售虚拟库存表(t_op_presale_virtual_stock)记录ID
     * @author gongjun
     */
    ServiceResp<Long> createPresaleVirtualStock(SVirtualStockVO stockVO);

    /**
     * 根据预售虚拟库存锁现货库存，在不同业务下有不同的解释（如：权益预售占用/锁现货）
     * <p>在现货到的时候，按到货数量以及虚拟库存得出的需求量将现货按订单预占用形式锁定（预占用）</p>
     * <p>内部主要通过调用{@link SStockService#prepare()}, {@link SStockService#incrementPreparedQuantity(SStockPrepareQuantityChangeDTO)}
     * 以及{@link SStockService#decrementPreparedQuantity(SStockPrepareQuantityChangeDTO)}接口实现</p>
     * <p></p>
     * <p>影响的表：</p>
     * <pre>
     * t_wh_inv_occupy
     * t_wh_inv_prepared_occupy_record
     * </pre>
     * <p>业务场景（权益预售到货）：</p>
     * <pre>
     * 当权益预售的现货到货的时候调用该方法，根据到货的数量以及权益预售需要的数量锁住现货库存。
     * 第一次锁的时候，会在t_wh_inv_occupy表中插入一条占用数据（预占用类型）。
     * 从第二次开始（已有t_wh_inv_occupy表记录），则会在t_wh_inv_prepared_occupy_record表中插入一条数量变动记录，
     * 该表记录可以为正值，也可以为负值。
     * 而且要确保t_wh_inv_prepared_occupy_record中数量的SUM的值要和t_wh_inv_occupy中对应记录的数量一致。
     * </pre>
     *
     * @param virtualOccupyDTO 预售虚占用参数，{@link SLockSpotStockDTO}实例
     * @return 预售虚占用结果，{@link SLockSpotStockResultVO}实例
     * @author gongjun
     */
    ServiceResp<SLockSpotStockResultVO> lockSpotStock(SLockSpotStockDTO virtualOccupyDTO);

    /**
     * 获取当前根据预售虚拟库存锁定的现货库存数量
     * <p>该方法获取的数量为当前的已锁现货数量，不包括已释放的现货数量，非累计</p>
     *
     * @param lockedRealStockQuantityDTO 已锁现货库存查询参数，{@link SLockedPromptStockQueryDTO}实例
     * @return 当前根据预售虚拟库存锁定的现货库存数量
     * @author gongjun
     */
    Integer getLockedRealStockQuantity(SLockedPromptStockQueryDTO lockedRealStockQuantityDTO);

    /**
     * 释放根据预售虚拟库存锁定的现货库存，在不同业务下有不同解释（如：释放权益预售占用/已锁现货）
     * <p>影响的表：</p>
     * <pre>
     * t_wh_inv_occupy
     * t_wh_inv_prepared_occupy_record
     * </pre>
     * <p>业务场景（释放权益预售占用）：</p>
     * <pre>
     * 在t_wh_inv_prepared_occupy_record表中插入一条数量为负的记录，并更新t_wh_inv_occupy中对应记录的数量
     * 而且要确保t_wh_inv_prepared_occupy_record中数量的SUM的值要和t_wh_inv_occupy中对应记录的数量一致
     * </pre>
     *
     * @return 释放已锁现货库存的结果VO类，{@link SReleaseSpotStockResultVO}实例
     * @author gongjun
     */
    ServiceResp<SReleaseSpotStockResultVO> releaseLockedSpotStock();



    ServiceResp updateVirtualTotalQuantity(SPresaleVirtualQuantityChangeDTO dto);
}
