package com.thebeastshop.payment.enums;

import com.thebeastshop.common.enums.CodeEnum;

import java.util.Arrays;
import java.util.List;

/**
 * 支付渠道枚举
 *
 * @author Paul-xiong
 * @date 2017年11月9日
 * @description 支付渠道
 */
public enum PPaymentChannelGroupEnum implements CodeEnum<Integer> {
    // 支付宝退款渠道，注：关联的paymentChannels支付渠道是有序的。
    ALIPAY(1, "支付宝",
            PPaymentChannelEnum.ALIPAY_MOBILE,
            PPaymentChannelEnum.ALIPAY_DIRECT,
            PPaymentChannelEnum.ALIPAY_SCAN,
            PPaymentChannelEnum.ALIPAY_WAP_DIRECT,
            PPaymentChannelEnum.ALIPAY_USER_SCAN,
            PPaymentChannelEnum.ALI_BANK,
            PPaymentChannelEnum.ALI_CROSS_BORDER_MOBILE,
            PPaymentChannelEnum.ALI_CROSS_BORDER_DIRECT,
            PPaymentChannelEnum.ALI_WAP_CROSS_BORDER_DIRECT
    ),
    // 微信退款渠道，注：关联的paymentChannels支付渠道是有序的。
    WXPAY(2, "微信",
            PPaymentChannelEnum.WEIXIN_MOBILE,
            PPaymentChannelEnum.WEIXIN_MINI_PROGRAM,
            PPaymentChannelEnum.WEIXIN_SCAN,
            PPaymentChannelEnum.WEIXIN_OFFICIAL_ACCOUNT,
            PPaymentChannelEnum.WEIXIN_USER_SCAN,
            PPaymentChannelEnum.WEIXIN_DIRECT
    ),
    CASH(3, "现金", PPaymentChannelEnum.CASH),
    DEBIT_CARD(4, "借记卡", PPaymentChannelEnum.DEBIT_CARD),
    CREDIT_CARD(5, "信用卡", PPaymentChannelEnum.CREDIT_CARD),
    TRANSFER(6, "转账", PPaymentChannelEnum.TRANSFER),
    YCK(7, "预存款", PPaymentChannelEnum.YCK),
    YEZHU(8, "业主", PPaymentChannelEnum.YEZHU),
    GIFT_CARD(9, "礼品卡", PPaymentChannelEnum.GIFT_CARD),
    INTEGRAL(10, "积分", PPaymentChannelEnum.INTEGRAL),
    CMB(11, "招商银行", PPaymentChannelEnum.CMB),
    MARKET_COUPON(12, "商场券",PPaymentChannelEnum.MARKET_COUPON),
    QUICK_PASS_SCAN(13, "银联云闪付",PPaymentChannelEnum.QUICK_PASS_SCAN),
    FUND(14, "基金",PPaymentChannelEnum.FUND),
    ELEME(15, "饿了么",PPaymentChannelEnum.ELEME),
    TM_SHOPPING_FUND(16, "天猫购物金", PPaymentChannelEnum.TM_SHOPPING_FUND),
    TM_INTERESTS_FUND(17, "天猫权益金", PPaymentChannelEnum.TM_INTERESTS_FUND),
    SHOPPING_MALL_APP(18, "商场小程序", PPaymentChannelEnum.SHOPPING_MALL_APP),
    MERCHANTS_PAY_MARKETING_DISCOUNT(19, "商家支付营销优惠", PPaymentChannelEnum.MERCHANTS_PAY_MARKETING_DISCOUNT),
    GROUP_BUYING_VOUCHER(20, "团购券", PPaymentChannelEnum.GROUP_BUYING_VOUCHER),
    WORK_WEIXIN_TRANSFER(21, "企业微信转账", PPaymentChannelEnum.WORK_WEIXIN_TRANSFER),
    PAYPAL(22, "PayPal", PPaymentChannelEnum.PAYPAL),
    MANUAL(23, "Manual", PPaymentChannelEnum.MANUAL),
    VISA(24, "Visa", PPaymentChannelEnum.VISA),
    MASTERCARD(25, "MasterCard", PPaymentChannelEnum.MASTERCARD),
    AMERICAN_EXPRESS(26, "American Express", PPaymentChannelEnum.AMERICAN_EXPRESS),
    LIANLIAN_PAY(27, "LianLian Pay", PPaymentChannelEnum.LIANLIAN_PAY)
    ;

    private Integer id;

    private String name;

    /**
     * 关联的支付渠道
     */
    private List<PPaymentChannelEnum> paymentChannels;

    PPaymentChannelGroupEnum(Integer id, String name, PPaymentChannelEnum... paymentChannels) {
        this.id = id;
        this.name = name;
        if (paymentChannels == null || paymentChannels.length <= 0) {
            throw new IllegalArgumentException("退款关联的支付渠道不能为空！");
        }
        this.paymentChannels = Arrays.asList(paymentChannels);
    }

    public Integer getCode() {
        return id;
    }

    public String getName() {
        return name;
    }

    public List<PPaymentChannelEnum> getPaymentChannels() {
        return paymentChannels;
    }

    /**
     * 通过code获取退款渠道
     *
     * @param code
     * @return
     */
    public static PPaymentChannelGroupEnum getEnumByCode(Integer code) {
        for (PPaymentChannelGroupEnum e : PPaymentChannelGroupEnum.values()) {
            if (e.getCode().equals(code)) {
                return e;
            }
        }
        return null;
    }

    /**
     * 通过支付渠道获取退款渠道
     *
     * @param paymentChannelCode
     * @return
     */
    public static PPaymentChannelGroupEnum getEnumByPaymentChannelCode(Integer paymentChannelCode) {
        PPaymentChannelEnum paymentChannel = PPaymentChannelEnum.getEnumByCode(paymentChannelCode);
        for (PPaymentChannelGroupEnum e : PPaymentChannelGroupEnum.values()) {
            if (e.getPaymentChannels().contains(paymentChannel)) {
                return e;
            }
        }
        return null;
    }

}


