package com.thebeastshop.mbgw.sdk.client;

import com.alibaba.fastjson.JSON;
import com.thebeastshop.mbgw.sdk.config.MbGwSdkConfig;
import com.thebeastshop.mbgw.sdk.request.MbGwRequest;
import com.thebeastshop.mbgw.sdk.response.MbGwResponse;
import com.thebeastshop.mbgw.sdk.util.MbGwHttpUtil;
import com.thebeastshop.mbgw.sdk.util.MbGwLogUtil;
import com.thebeastshop.mbgw.sdk.util.MbGwSignUtil;
import org.slf4j.Logger;

import java.util.HashMap;
import java.util.Map;

public class MbGwClientImpl implements MbGwClient {

    private static final Logger log = MbGwLogUtil.getLogger(MbGwClientImpl.class);

    @Override // 实现接口方法
    public <T> MbGwResponse<T> execute(MbGwRequest request, Class<T> dataType) {
        long startTime = System.currentTimeMillis();
        String apiName = request.getApiName();
        log.info("开始调用API：{}（请求ID：{}）", apiName, System.currentTimeMillis());

        try {
            // 1. 拆分文本参数和文件参数
            Map<String, Object> textParams = new HashMap<>();
            Map<String, String> fileParams = new HashMap<>();
            boolean hasFile = splitParams(request.getParams(), textParams, fileParams);
            log.info("API[{}]参数拆分完成：文本参数{}个，文件参数{}个",
                    apiName, textParams.size(), fileParams.size());

            // 2. 构建基础请求参数（含公共参数）
            Map<String, Object> baseParams = buildBaseParams(request, textParams);
            String requestJson = JSON.toJSONString(baseParams);
            log.info("API[{}]请求参数：{}", apiName, requestJson);

            // 3. 生成签名
            String sign = MbGwSignUtil.generateSign(requestJson);
            log.info("API[{}]签名生成完成：{}", apiName, sign.substring(0, 10) + "...");

            // 4. 发送请求（根据是否有文件选择请求方式）
            String responseJson;
            if (hasFile) {
                Map<String, String> formTextParams = new HashMap<>();
                baseParams.forEach((k, v) -> formTextParams.put(k, v.toString()));
                responseJson = MbGwHttpUtil.sendFormPost(formTextParams, fileParams, sign);
                log.info("API[{}]文件上传请求发送完成", apiName);
            } else {
                responseJson = MbGwHttpUtil.sendJsonPost(requestJson, sign);
                log.info("API[{}]JSON请求发送完成", apiName);
            }
            log.info("API[{}]原始响应：{}", apiName, responseJson);

            // 5. 解析响应
            MbGwResponse<T> response = MbGwResponse.parse(responseJson, dataType);
            long costTime = System.currentTimeMillis() - startTime;
            if (response.isSuccess()) {
                log.info("API[{}]调用成功（耗时：{}ms，响应码：{}）",
                        apiName, costTime, response.getCode());
            } else {
                log.warn("API[{}]调用失败（耗时：{}ms，响应码：{}，信息：{}）",
                        apiName, costTime, response.getCode(), response.getMessage());
            }
            return response;

        } catch (Exception e) {
            long costTime = System.currentTimeMillis() - startTime;
            log.error("API[{}]调用异常（耗时：{}ms）", apiName, costTime, e);
            throw e;
        }
    }

    /**
     * 拆分参数为文本参数和文件参数
     * @param params 原始参数
     * @param textParams 输出：文本参数（非文件类型）
     * @param fileParams 输出：文件参数（路径以@开头的参数）
     * @return 是否包含文件参数
     */
    private boolean splitParams(Map<String, Object> params,
                                Map<String, Object> textParams,
                                Map<String, String> fileParams) {
        boolean hasFile = false;
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();

            if (value instanceof String && ((String) value).startsWith("@")) {
                // 处理文件参数：去掉@前缀，获取真实路径
                String filePath = ((String) value).substring(1);
                fileParams.put(key, filePath);
                hasFile = true;
            } else {
                // 处理文本参数
                textParams.put(key, value);
            }
        }
        return hasFile;
    }

    /**
     * 构建基础请求参数（包含公共参数）
     */
    private Map<String, Object> buildBaseParams(MbGwRequest request, Map<String, Object> textParams) {
        Map<String, Object> baseParams = new HashMap<>();
        baseParams.put("api", request.getApiName());
        baseParams.put("appkey", MbGwSdkConfig.getApiKey());
        baseParams.put("accesstoken", request.getAccessToken());
        baseParams.put("data", textParams);
        baseParams.put("timestamp", System.currentTimeMillis() / 1000); // 秒级时间戳
        return baseParams;
    }
}
