package com.thebeastshop.mbgw.sdk.request;


import java.util.HashMap;
import java.util.Map;

/**
 * 马帮ERP API 请求模型
 * 封装接口名称、业务参数和访问令牌，支持普通参数和文件参数的添加
 */
public class MbGwRequest {
    // 接口名称（如：sys-get-currency-rate-list、sys-upload-file）
    private String apiName;
    // 业务参数集合（key：参数名，value：参数值）
    private Map<String, Object> params = new HashMap<>();
    // 访问令牌（可选，部分接口需要）
    private String accessToken = "";

    /**
     * 构造请求对象
     * @param apiName 接口名称
     */
    public MbGwRequest(String apiName) {
        this.apiName = apiName;
    }

    /**
     * 添加普通业务参数
     * @param key 参数名
     * @param value 参数值（支持String、Number、Boolean等基本类型）
     * @return 当前请求对象（链式调用）
     */
    public MbGwRequest addParam(String key, Object value) {
        if (key == null || key.isEmpty()) {
            throw new IllegalArgumentException("参数名不能为空");
        }
        params.put(key, value);
        return this;
    }

    /**
     * 添加文件参数（内部自动标记文件类型）
     * @param key 文件参数名（如：mbFile）
     * @param filePath 本地文件绝对路径或相对路径
     * @return 当前请求对象（链式调用）
     */
    public MbGwRequest addFileParam(String key, String filePath) {
        if (filePath == null || filePath.isEmpty()) {
            throw new IllegalArgumentException("文件路径不能为空");
        }
        // 用@前缀标记文件参数，便于后续拆分（内部逻辑识别）
        params.put(key, "@" + filePath);
        return this;
    }

    // Getter和Setter
    public String getApiName() {
        return apiName;
    }

    public void setApiName(String apiName) {
        this.apiName = apiName;
    }

    public Map<String, Object> getParams() {
        return new HashMap<>(params); // 返回副本，避免外部修改
    }

    public String getAccessToken() {
        return accessToken;
    }

    /**
     * 设置访问令牌（可选）
     * @param accessToken 令牌字符串
     */
    public void setAccessToken(String accessToken) {
        this.accessToken = accessToken == null ? "" : accessToken;
    }

    @Override
    public String toString() {
        return "MbGwRequest{" +
                "apiName='" + apiName + '\'' +
                ", params=" + params +
                ", accessToken='" + (accessToken.isEmpty() ? "无" : "***") + '\'' +
                '}';
    }
}
