package com.thebeastshop.common.utils;

import jxl.*;
import jxl.read.biff.BiffException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.core.convert.ConversionService;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.validation.Validator;

import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

@Component("parseExcelComponent")
@Scope(value = "request", proxyMode = ScopedProxyMode.TARGET_CLASS)
public class ParseExcelComponent<T> {

	private Class<T> clazz;

	private InputStream is;

	private List<ObjectError> errorMessages = new ArrayList<ObjectError>();

	private List<String> warnMessages = new ArrayList<String>();;

	private Set<String> headerColumn = new HashSet<String>();

	private Set<String> warnMessagesSet = new HashSet<>();

	private Map<Integer, Field> columnMap = new HashMap<Integer, Field>();

	private Map<String, String> fieldNameMap = new HashMap<String, String>();

	private List<T> availableResults = new ArrayList<T>();

	private List<T> allResults = new ArrayList<T>();

	private List<T> failureResults = new ArrayList<T>();

	private Method setErrorMsgMethod;

	private Method getErrorMsgMethod;

	private Method setColumnNumberMethod;
	@Autowired
	private Validator beanVidator;

	@Autowired
	private ConversionService myConversionService;

	public void parse(InputStream is, Class<T> clazz) {
		this.is = is;
		this.clazz = clazz;
		validateExcel();
	}

	public void parse(File file, Class<T> clazz) {
		if (file == null) {
			throw new IllegalArgumentException("the file is null!");
		}
		try {
			this.is = new FileInputStream(file);
			this.clazz = clazz;
			validateExcel();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	public void parse(String fileName, Class<T> clazz) {
		if (fileName == null || fileName.trim().length() < 1)
			throw new IllegalArgumentException("the file path is empty");
		File file = new File(fileName);
		try {
			this.is = new FileInputStream(file);
			this.clazz = clazz;
			validateExcel();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	public List<T> getAvailableResults() {
		return availableResults;
	}

	public List<ObjectError> getErrorMessages() {
		return errorMessages;
	}
	public String getErrorInfo(){
		StringBuffer stringBuffer = new StringBuffer();
		if(errorMessages != null && errorMessages.size() > 0){
			for(ObjectError error :errorMessages){
				stringBuffer.append(error.getDefaultMessage());
				stringBuffer.append("</br>");
			}
		}
		return stringBuffer.toString();
	}
	public List<T> getFailureResults() {
		return failureResults;
	}

	public List<T> getAllResults() {
		return allResults;
	}

	public boolean hasErrors() {
		return errorMessages.size() > 0 ? true : false;
	}

	public List<String> getWarnMessages() {
		return warnMessages;
	}

	public boolean hasWarns() {
		return warnMessages.size() > 0 ? true : false;
	}

	private void validateExcel() {
		ExcelTemplate et = clazz.getAnnotation(ExcelTemplate.class);
		if (et == null) {
			throw new RuntimeException("the class is not a excel template data");
		}
        try {
            if (is.available() > 2*1024 * 1024) { // 2MB限制
                throw new RuntimeException("文件大小不能超过2MB！");
            }
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }
        try {
			WorkbookSettings workbooksetting = new WorkbookSettings();
			workbooksetting .setCellValidationDisabled(true);
			Workbook wb = Workbook.getWorkbook(is,workbooksetting);
			Sheet sheet = wb.getSheet(0);
			if (sheet.getRows() <= 1) {
				errorMessages.add(new ObjectError(clazz.getName(), "Excel文件格式不正确，请严格按照给定的格式生成excel文件!"));
				return;
			}
			Cell[] firstRow = sheet.getRow(0);
			if (firstRow.length < 1) {
				errorMessages.add(new ObjectError(clazz.getName(), "Excel文件没有第一行，请严格按照给定的格式生成excel文件!"));
				return;
			}
			checkExcelHeader(firstRow);
			if (errorMessages.size() > 0) {
				return;
			}
			if (headerColumn.size() < 1) {
				errorMessages.add(new ObjectError(clazz.getName(), "Excel文件第一行的列名与模板完全不符,请严格按照给定的格式生成excel文件!"));
				return;
			}
			// 初始化注释ErrorMessage字段的set和get方法和columnNumber字段的set方法
			initErrorMessageAndColumnNumberMehtod();
			// 把excel转换为javabean对象，然后再对每个属性进行验证
			convertToJavaBean(sheet);
			// 验证生成的javabean
			int rowTip = 1;
			for (T t : allResults) {
				rowTip++;
				validate(t, rowTip);
			}
			// 检查注释ExcelColumn中unique为true的field
			checkUniqueConstrain();

		} catch (BiffException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SecurityException e) {
			e.printStackTrace();
		}
	}

	private void checkExcelHeader(Cell[] firstRowCells) {
		Class<? super T> temp = clazz;
		List<Field> fieldsList = new ArrayList<Field>();
		while (temp != null && !temp.equals(Object.class) && temp.getAnnotation(ExcelTemplate.class) != null) {
			Field[] fs = temp.getDeclaredFields();
			for (int x = 0; x < fs.length; x++) {
				ExcelColumn ec = fs[x].getAnnotation(ExcelColumn.class);
				if (ec == null) {
					continue;
				}
				fieldsList.add(fs[x]);
			}
			temp = temp.getSuperclass();
		}
		for (int i = 0; i < firstRowCells.length; i++) {
			String content = firstRowCells[i].getContents();
			if (content == null || content.trim().length() < 1) {
				continue;
			}
			content = content.trim();
			for (int j = 0; j < fieldsList.size(); j++) {
				ExcelColumn ec = fieldsList.get(j).getAnnotation(ExcelColumn.class);
				if (ec == null) {
					continue;
				}
				String columnName = ec.name().equals("") ? fieldsList.get(j).getName() : ec.name();
				fieldNameMap.put(fieldsList.get(j).getName(), ec.name());
				if (content.equals(columnName)) {
					if (!headerColumn.add(columnName)) {
						errorMessages.add(new ObjectError(clazz.getName(), "在模板中发现有重复的列名" + columnName));
					} else {
						columnMap.put(i, fieldsList.get(j));
					}
				}
			}
		}
	}

	private void convertToJavaBean(Sheet sheet) {
		try {
			int row = sheet.getRows();
			for (int i = 1; i < row; i++) {
				T obj = clazz.newInstance();
				Cell[] cells = sheet.getRow(i);
				int nullCount = 0;
				for (Integer j : columnMap.keySet()) {
					Field field = columnMap.get(j);
					if (field == null) {
						nullCount++;
						continue;
					}
					// 可能头的列数会大于下面内容的列数，所以做如下处理
					if (j < cells.length) {
						String chineseName = fieldNameMap.get(field.getName()).length() < 1 ? field.getName() : fieldNameMap.get(field.getName());
						if (CellType.EMPTY.equals(cells[j].getType()) || CellType.ERROR.equals(cells[j].getType())) {
							nullCount++;
							continue;
						}
						Cell cell = cells[j];
						CellType cellType = cell.getType();
						if(CellType.NUMBER_FORMULA.equals(cellType)
								|| CellType.STRING_FORMULA.equals(cellType)
								|| CellType.BOOLEAN_FORMULA.equals(cellType)
								|| CellType.DATE_FORMULA.equals(cellType)
								|| CellType.FORMULA_ERROR.equals(cellType)){
							warnMessages.add("第" + (cell.getRow()+1) + "行中," + chineseName + "存在公式，无法上传");
							warnMessagesSet.add(chineseName  + (i+1));
							errorMessages.add(new
									FieldError(clazz.getName(),field.getName(),"第"+(cell.getRow()+1)+"行中,"+chineseName+"存在公式，无法上传"));
							continue;
						}
						String content = cell.getContents().trim();
						if (BigDecimal.class.getName().equals(field.getType().getName())
						    	&& CellType.NUMBER.equals(cellType)) {
							NumberCell numberCell = (NumberCell) cell;
							content = String.valueOf(numberCell.getValue());
						}
						if (content.length() > 0) {
							Method method = clazz.getMethod(generateSetMethod(field.getName()), field.getType());
							// 如果为string类型则不做转型
							if (!String.class.getName().equals(field.getType().getName())) {
								try {
									// excel里面是date类型，对应模板里也是date类型的情况
									if (Date.class.getName().equals(field.getType().getName())) {
										CellType cellTypes = cells[j].getType();
										String type = cellTypes.toString();
										TimeZone zone = TimeZone.getTimeZone("GMT");
										SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
										sdf.setTimeZone(zone);
										if(type.equalsIgnoreCase("DATE")){
										   DateCell dc = (DateCell) cells[j];
										   String context = sdf.format(dc.getDate());
										   TimeZone local = TimeZone.getDefault();
										   sdf.setTimeZone(local);
										   Date date = sdf.parse(context);
										   method.invoke(obj, date);
										}else{
										   Date date = DateUtil.parse(content,DateUtil.DEFAULT_DATETIME_FORMAT);
											 //date 日期格式化（yyyy-MM-dd）
											 if(date == null) {
												 date = DateUtil.parse(content, DateUtil.DEFAULT_DATE_FORMAT);
											 }
											//date 日期格式化（yyyy/MM/dd）
											 if (date == null) {
												 date = DateUtil.parse(content, "yyyy/MM/dd");
											 }
										   method.invoke(obj, date);
										}
									} else {
										method.invoke(obj, myConversionService.convert(content, field.getType()));
									}
								} catch (Exception e) {
									warnMessages.add("第" + (i + 1) + "行中," + chineseName + "格式不正确");
									errorMessages.add(new
											FieldError(clazz.getName(),field.getName(),"第"+i+"行中,"+chineseName+"格式不正确"));
								}
							} else {
								method.invoke(obj, content);
								// System.out.println("~~~~~~~~~~~"+method.getName());
								// Method
								// method=ReflectionUtils.findMethod(clazz,
								// field.getName());
							}
						} else {
							nullCount++;
						}
					} else {
						nullCount++;
					}
				}
				if (columnMap.size() > nullCount) {
					setExcelColumnNumber(obj, i + 1);
					allResults.add(obj);
				}
			}
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private String getCellContent(Cell cell){
		if(CellType.NUMBER_FORMULA.equals(cell.getType())
				|| CellType.STRING_FORMULA.equals(cell.getType())
				|| CellType.BOOLEAN_FORMULA.equals(cell.getType())
				|| CellType.DATE_FORMULA.equals(cell.getType())
				|| CellType.FORMULA_ERROR.equals(cell.getType())){
			throw new RuntimeException(String.format("%s行%s列存在公式，无法上传",cell.getRow(),cell.getColumn()));
		}else{
			return cell.getContents().trim();
		}
	}

	// 验证通过添加到availableResults否则添加到failureResults
	@SuppressWarnings("unchecked")
	private void validate(T obj, int row) {
		BindException errors = new BindException(obj, clazz.getSimpleName());
		beanVidator.validate(obj, errors);
		if (errors.hasErrors()) {
			for (FieldError error : (List<FieldError>) errors.getFieldErrors()) {

				if (fieldNameMap.get(error.getField()) == null) {
					continue;
				}
				String chineseName = fieldNameMap.get(error.getField()).length() < 1 ? error.getField() : fieldNameMap.get(error.getField());
				//已经校验格式有问题的单元格，无需再check是否为空
				if(!warnMessagesSet.contains(chineseName + row)) {
					errorMessages.add(new FieldError(error.getObjectName(), error.getField(), "第" + (row) + "行中," + chineseName
							+ error.getDefaultMessage()));
					setErrorMessage(obj, chineseName + error.getDefaultMessage());
				}
			}
			failureResults.add(obj);
		} else {
			availableResults.add(obj);
		}
	}

	private void checkUniqueConstrain() {
		// 得到unique为true的field
		Set<Field> fields = new HashSet<Field>();
		for (int i : columnMap.keySet()) {
			Field field = columnMap.get(i);
			ExcelColumn ec = field.getAnnotation(ExcelColumn.class);
			if (ec == null) {
				continue;
			}
			if (ec.unique() == true) {
				fields.add(field);
			}
		}
		if (fields.size() < 1) {
			return;
		}
		try {
			List<T> needToRemove = new ArrayList<T>();
			for (Field f : fields) {
				Method method = clazz.getMethod(generateGetMethod(f.getName()));
				for (int i = 0; i < availableResults.size(); i++) {
					T t = availableResults.get(i);
					Object obj = method.invoke(t);
					int rowDis = 2;
					boolean hasIt=false;
					String chineseName = fieldNameMap.get(f.getName()).length() < 1 ? f.getName() : fieldNameMap.get(f.getName());
					for (int j = 0; j < availableResults.size(); j++) {
						T sub = availableResults.get(j);
						if (j != i && !needToRemove.contains(sub)) {
							Object sobj = method.invoke(sub);
							if (obj != null && obj.equals(sobj)) {
								errorMessages.add(new FieldError(clazz.getName(), "multiple_" + f.getName(), "第" + (i + rowDis) + "行和第"
										+ (j + rowDis) + "行中的" + chineseName + "数据重复!"));
								setErrorMessage(sub, chineseName + "数据重复!");
								needToRemove.add(sub);
								hasIt=true;
							}
						}
					}
					if(hasIt){
						needToRemove.add(t);
						setErrorMessage(t, chineseName + "数据重复!");
					}
				}
				if (needToRemove.size() > 0) {
					availableResults.removeAll(needToRemove);
					failureResults.addAll(needToRemove);
				}
				needToRemove.clear();
			}
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}
	}

	private void initErrorMessageAndColumnNumberMehtod() {
		Field[] fields = clazz.getDeclaredFields();
		try {
			for (int i = 0; i < fields.length; i++) {
				ErrorMessage em = fields[i].getAnnotation(ErrorMessage.class);
				ExcelColumnNumber ecn = fields[i].getAnnotation(ExcelColumnNumber.class);
				if (em != null && setErrorMsgMethod == null) {
					setErrorMsgMethod = clazz.getMethod(generateSetMethod(fields[i].getName()), fields[i].getType());
					getErrorMsgMethod = clazz.getMethod(generateGetMethod(fields[i].getName()));
				}
				if (ecn != null && setColumnNumberMethod == null) {
					setColumnNumberMethod = clazz.getMethod(generateSetMethod(fields[i].getName()), fields[i].getType());
				}
			}
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		}
	}

	private void setExcelColumnNumber(T obj, Integer number) {
		if (setColumnNumberMethod != null) {
			try {
				setColumnNumberMethod.invoke(obj, number);
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
		}
	}

	private void setErrorMessage(T obj, String message) {
		if (setErrorMsgMethod != null)
			try {
				String em = getErrorMessage(obj);
				if (em != null && em.trim().length() > 0) {
					setErrorMsgMethod.invoke(obj, em + ";" + message);
				} else {
					setErrorMsgMethod.invoke(obj, message);
				}
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
	}

	private String getErrorMessage(T obj) {
		if (getErrorMsgMethod != null)
			try {
				return (String) getErrorMsgMethod.invoke(obj);
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
		return null;
	}

	private String generateSetMethod(String fieldName) {
		if (fieldName == null || fieldName.length() < 1) {
			return "";
		}
		return "set" + fieldName.toUpperCase().charAt(0) + fieldName.substring(1);
	}

	private String generateGetMethod(String fieldName) {
		if (fieldName == null || fieldName.length() < 1) {
			return "";
		}
		return "get" + fieldName.toUpperCase().charAt(0) + fieldName.substring(1);
	}

	public static void main(String[] args) {

		System.out.println(new BigDecimal(3.1715));

		/*
		 * File file=new File("e:/0510.xls"); ApplicationContext context = new
		 * ClassPathXmlApplicationContext( new
		 * String[]{"classpath*:META-INF/spring/applicationContext*.xml"});
		 * Validator beanValidator=(Validator)context.getBean("beanValidator");
		 * ConversionService
		 * conversionService=(ConversionService)context.getBean
		 * ("conversionService"); ParseExcelComponent<MemberImportData> pe=new
		 * ParseExcelComponent<MemberImportData>(file,MemberImportData.class);
		 * pe.setBeanVidator(beanValidator);
		 * pe.setConversionService(conversionService); pe.validateExcel();
		 *
		 * for(String e:pe.getErrorMessages()){ System.out.println(e); }
		 */
	}

	// //only for test
	// public void setConversionService(ConversionService conversionService) {
	// this.conversionService = conversionService;
	// }
	//
	// //only for test
	// public void setBeanVidator(Validator beanVidator) {
	// this.beanVidator = beanVidator;
	// }
}
