package com.thebeastshop.kit.rocketmq.springboot;

import cn.hutool.core.util.ReflectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.thebeastshop.common.utils.BeanUtil;
import com.thebeastshop.kit.rocketmq.RDelay;
import com.thebeastshop.kit.rocketmq.RMessage;
import com.thebeastshop.kit.rocketmq.RTopicInvoker;
import com.thebeastshop.kit.rocketmq.RocketMQTopicException;
import com.thebeastshop.kit.rocketmq.Topic;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyContext;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.core.RocketMQListener;
import org.apache.rocketmq.spring.core.RocketMQPushConsumerLifecycleListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

public class RocketMQConsumer implements RocketMQListener<String>, RocketMQPushConsumerLifecycleListener {

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private final Map<String, RTopicInvoker> TOPIC_INVOKER_CACHE = new ConcurrentHashMap<>();

    @Override
    public void onMessage(String s) {
    }


    @Override
    public void prepareStart(DefaultMQPushConsumer consumer) {
        final Class<?> clazz = this.getClass();
        final Method[] methods =  ReflectUtil.getMethods(clazz);

        for (Method method : methods) {
            Class<?>[] paramTypes = method.getParameterTypes();
            Topic topicAnnotation = method.getAnnotation(Topic.class);
            if (topicAnnotation == null) {
                continue;
            }
            String topic = topicAnnotation.value();
            if (StringUtils.isBlank(topic)) {
                topic = "";
            } else {
                topic = topic.trim();
            }
            try {
                consumer.subscribe(topic, "*");
            } catch (MQClientException e) {
                throw new RuntimeException("[RocketMQ] 订阅Topic[\"" + topic + "\"; tags = \"*\"]失败: ", e);
            }
            final Class<?> paramType = paramTypes[0];
            TOPIC_INVOKER_CACHE.put(topic, new RTopicInvoker(paramType, method));
        }

        consumer.registerMessageListener((List<MessageExt> messages, ConsumeConcurrentlyContext context) -> {
            if (CollectionUtils.isNotEmpty(messages)) {
                for (MessageExt message : messages) {
                    try {
                        final String topic = message.getTopic();
                        RTopicInvoker invoker = TOPIC_INVOKER_CACHE.get(topic);
                        if (invoker == null) {
                            invoker = TOPIC_INVOKER_CACHE.get(topic);
                            if (invoker == null) {
                                throw new RocketMQTopicException(topic);
                            }
                        }
                        logMessage(message);
                        invoker.invoke(this, message);
                    } catch (Throwable th) {
                        log.error("[RocketMQ] 执行消费者任务时发生错误: ", th);
                        return ConsumeConcurrentlyStatus.RECONSUME_LATER;
                    }
                }
            }
            return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
        });
    }


    private void logMessage(MessageExt message) {
        StringBuilder content = new StringBuilder("[RocketMQ 消费者] 接受消息(")
                .append("id = ")
                .append(message.getProperty(RMessage.KEY_MSG_ID))
                .append(", topic = ")
                .append(message.getTopic());
        String tags = message.getTags();
        if (StringUtils.isNotEmpty(tags)) {
            content.append(", tags = ")
                    .append(tags);
        }
        String delay = RDelay.timeLevelToString(message.getDelayTimeLevel());
        if (StringUtils.isNotEmpty(delay)) {
            content.append(", delay = ")
                    .append(delay);
        }
        content.append("):\n\t");
        content.append(new String(message.getBody(), StandardCharsets.UTF_8));
        log.info(content.toString());
    }

}
