package com.thebeastshop.kit.rocketmq;

import com.thebeastshop.common.validation.Validation;
import com.thebeastshop.kit.id.UniqueIdGenerator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.client.exception.MQBrokerException;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.remoting.exception.RemotingException;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;

public class RClient {
    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private final RocketMQTemplate rocketMQTemplate;

    private static RClient rClient;

    private long defaultSendMessageTimeout = 3000;

    public static RClient getClient(final RocketMQTemplate rocketMQTemplate) {
        if (rClient == null) {
            rClient = new RClient(rocketMQTemplate);
        }
        return rClient;
    }

    public static RClient getClient() {
        return rClient;
    }


    public RClient(RocketMQTemplate rocketMQTemplate) {
        this.rocketMQTemplate = rocketMQTemplate;
    }

    public long getDefaultSendMessageTimeout() {
        return defaultSendMessageTimeout;
    }

    public void setDefaultSendMessageTimeout(long defaultSendMessageTimeout) {
        this.defaultSendMessageTimeout = defaultSendMessageTimeout;
    }

    public RocketMQTemplate getRocketMQTemplate() {
        return rocketMQTemplate;
    }

    public DefaultMQProducer getProducer() {
        return rocketMQTemplate.getProducer();
    }

    public void processException(Exception ex, OnException onException) {
        if (onException == null) {
            throw new RuntimeException(ex);
        }
        onException.onException(ex);
    }

    public SendResult syncSend(Message message, long timeout) {
        return syncSend(message, null, null, timeout);
    }

    private SendResult syncSend(Message message, OnSend onSend, OnException onException, long timeout) {
        try {
            SendResult sendResult = getProducer().send(
                    message,
                    timeout > 0 ? timeout : defaultSendMessageTimeout);
            if (onSend != null) {
                onSend.onSend(sendResult);
            }
            return sendResult;
        } catch (MQClientException | MQBrokerException | InterruptedException | RemotingException e) {
            processException(e, onException);
        }
        return null;
    }


    private void asyncSend(Message message, SendCallback sendCallback, long timeout) {
        try {
            getProducer().send(
                    message,
                    sendCallback,
                    timeout > 0 ? timeout : defaultSendMessageTimeout);
        } catch (MQClientException | RemotingException | InterruptedException e) {
            throw new RuntimeException(e);
        }
    }


    public <T> SendResult syncSend(final RMessage<T> message) {
        Validation.paramNotNull(message, "message 为空");
        beforeSend(message);
        final Message rocketMessage = message.toRocketMessage();
        logMessage(message, rocketMessage);
        return syncSend(
                rocketMessage,
                message.onSend(),
                message.onException(),
                message.timeout());
    }


    public <T> void asyncSend(final RMessage<T> message) {
        Validation.paramNotNull(message, "message 为空");
        beforeSend(message);
        final Message rocketMessage = message.toRocketMessage();
        logMessage(message, rocketMessage);
        asyncSend(
                rocketMessage,
                new SendCallback() {
                    @Override
                    public void onSuccess(SendResult sendResult) {
                        OnSend onSend = message.onSend();
                        if (onSend != null) {
                            onSend.onSend(sendResult);
                        }
                    }
                    @Override
                    public void onException(Throwable throwable) {
                        OnException onException = message.onException();
                        if (onException != null) {
                            onException.onException(throwable);
                        }
                    }
                },
                message.timeout());
    }

    private <T> void beforeSend(RMessage<T> message) {
        long id = UniqueIdGenerator.generateId();
        message.msgId(id + "");
        if (message.beforeSend() != null) {
            message.beforeSend().beforeSend(message);
        }
    }


    private <T> void logMessage(RMessage<T> message, Message rocketMessage) {
        if (message.logEnabled()) {
            StringBuilder content = new StringBuilder("[RocketMQ 生产者] 发送消息(")
                    .append("id = ")
                    .append(message.msgId())
                    .append(", topic = ")
                    .append(rocketMessage.getTopic());

            if (StringUtils.isNotEmpty(message.tags())) {
                content.append(", tags = ")
                        .append(message.tags());
            }

            String delay = RDelay.timeLevelToString(rocketMessage.getDelayTimeLevel());
            if (StringUtils.isNotEmpty(delay)) {
                content.append(", delay = ")
                        .append(delay);
            }

            content.append("):\n\t");
            content.append(new String(rocketMessage.getBody(), StandardCharsets.UTF_8));
            log.info(content.toString());
        }
    }

}
