package com.thebeastshop.kit.prop;

import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigChangeListener;
import com.ctrip.framework.apollo.ConfigService;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.google.common.collect.Lists;
import com.thebeastshop.kit.prop.annotation.DynamicPropAccessor;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.ConfigurableEnvironment;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2019-03-07 11:32
 */
public class PropConfig {

    private static final Logger log = LoggerFactory.getLogger(PropConfig.class);

    private static final AtomicBoolean INITIALIZED = new AtomicBoolean(false);

    private static final AtomicBoolean PRINTED = new AtomicBoolean(false);

    private static final AtomicBoolean MERGED = new AtomicBoolean(false);

    private static final String ENV_KEY = "env";

    private static final String APP_ID = "app.id";

    private static final String APOLLO_CLUSTER = "apollo.cluster";

    private static final String NAMESPACES_KEY = "app.namespaces";

    private static final String NAMESPACE_APPLICATION = "application";

    public static final String ENV_LOCAL = "local";

    public static final String ENV_TEST = "test";

    public static final String ENV_PRE = "pre";

    public static final String ENV_PROD = "prod";

    private static final String APOLLO_FAT = "fat";

    private static final String APOLLO_UAT = "uat";

    private static final String APOLLO_PRO = "pro";

    private volatile static PropConfig propConfig;

    private volatile ConfigurableEnvironment environment;

    private Properties props = new Properties();

    private ConfigMeta meta;

    private Map<String, String> namespaceMap = new HashMap<>();

    private Map<String, Map<String, Map<String, PropChangeCallback>>> propChangeCallbackMap = new HashMap<>();

    private static Map<String, Map<String, DynamicPropAccessor>> propAccessorsMap = new HashMap<>();

    private List<String> propList;

    private static List<PropPatternProcessor> patternProcessorList = new ArrayList<>();
    static {
        ServiceLoader<PropPatternProcessor> loader = ServiceLoader.load(PropPatternProcessor.class);
        if(loader != null){
            Iterator<PropPatternProcessor> it = loader.iterator();
            while(it.hasNext()){
                PropPatternProcessor processor = it.next();
                patternProcessorList.add(processor);
            }
        }
    }

    public PropConfig() {
        log.info("init Configuration");
    }

    /**
     * 获取resources/META-INF/app.properties中的配置信息
     * @param name
     * @return
     */
    public static String getMetaAppProperty(String name) {
        return ServiceUtil.getAppProperty(name);
    }

    /**
     * 添加属性匹配处理器
     * @param patternProcessor
     */
    public static void addPatternProcessor(PropPatternProcessor patternProcessor) {
        patternProcessorList.add(patternProcessor);
    }

    /**
     * 获取当前App的ID
     * @return
     */
    public static String getAppId() {
        return ServiceUtil.getAppId();
    }


    public static PropConfig getInstance() {
        if (propConfig == null) {
            synchronized (PropConfig.class) {
                if (propConfig == null) {
                    propConfig = new PropConfig();
                    propConfig.initializeProperties(new Properties());
                }
            }
        }
        return propConfig;
    }

    public void mergeLocalConfigSource(Map<?, ?> map) {
        if (MERGED.compareAndSet(false, true)) {
            props.putAll(map);
        }
    }

    public static String getProperties(String key){
        return getInstance().getProperty(key);
    }

    public static String getProperties(String key, String defaultValue) {
        return getInstance().getProperty(key, defaultValue);
    }

    public static Object setProperties(String key, String value) {
        return getInstance().setProperty(key, value);
    }


    public void setBootEnvironment(ConfigurableEnvironment environment) {
        this.environment = environment;
    }

    /**
     * 根据优先级获取配置信息
     * 优先级: System Property > 本地配置文件 >  服务端文件配置 > METE-INF/app.properties > 环境变量
     * @param properties
     * @param key
     * @return
     */
    private String getPrioritizedProperty(Properties properties, String key, String defaultValue) {
        String value = System.getProperty(key);
        if (StringUtils.isEmpty(value)) {
            if (props != null) {
                Object obj = props.get(key);
                if (obj != null) return obj.toString();
            }
            if (StringUtils.isEmpty(value)) {
                value = properties.getProperty(key);
                if (StringUtils.isEmpty(value)) {
                    value = ServiceUtil.getServerSideAppProperty(key);
                    if (StringUtils.isEmpty(value)) {
                        value = ServiceUtil.getServerSideProperty(key);
                        if (StringUtils.isEmpty(value)) {
                            value = getMetaAppProperty(key);
                            if (StringUtils.isEmpty(value)) {
                                value = System.getenv(key);
                                if (StringUtils.isEmpty(value)) {
                                    value = defaultValue;
                                }
                            }
                        }
                    }
                }
            }
        }
        return value;
    }


    private String getPrioritizedProperty(Properties properties, String key) {
        return getPrioritizedProperty(properties, key, null);
    }




    /**
     * 获取运行环境
     * @return
     */
    public static String getEnv(Properties properties) {
        String env = getInstance().getPrioritizedProperty(properties, ENV_KEY);
        if (StringUtils.isEmpty(env)) {
            return ENV_LOCAL;
        }
        return getApolloEnvName(env);
    }

    /**
     * 获取运行环境
     * @return
     */
    public static String getEnv() {
        return getEnv(System.getProperties());
    }


    /**
     * 获取在Apollo中的集群名称
     * @param properties
     * @return
     */
    public static String getApolloCluster(Properties properties) {
        String cluster = getInstance().getPrioritizedProperty(properties, APOLLO_CLUSTER);
        if (StringUtils.isNotBlank(cluster)) {
            return cluster.trim();
        }
        return null;
    }

    /**
     * 获取在Apollo中的集群名称
     * @return
     */
    public static String getApolloCluster() {
        return getApolloCluster(System.getProperties());
    }


    /**
     * 获取命名空间
     * @param meta
     * @param properties
     * @return
     */
    public List<String> getNamespaces(ConfigMeta meta, Properties properties) {
        String namespaces = getPrioritizedProperty(properties, NAMESPACES_KEY);
        List<String> namespacesList = new ArrayList<>();
        if (StringUtils.isEmpty(namespaces)) {
            return namespacesList;
        }
        meta.namespaces = namespaces.trim();
        this.props.put(NAMESPACES_KEY, namespaces);
        if (StringUtils.isNotEmpty(namespaces)) {
            String[] nsArr = namespaces.split(",");
            for (String ns : nsArr) {
                ns = ns.trim();
                if (!NAMESPACE_APPLICATION.equals(ns)) {
                    namespacesList.add(ns);
                }
            }
        }
        meta.namespaceList = namespacesList;
        return namespacesList;
    }


    Properties getProps() {
        return props;
    }

    /**
     * 配置元信息
     */
    private static class ConfigMeta {
        public String metaName;
        public String metaUrl;
        public String namespaces;
        public List<String> namespaceList;
        public Map<String, Properties> properties = new LinkedHashMap<>();
        public Properties otherProperties = new Properties();
    }

    /**
     * 获取配置中心地址
     * @return
     */
    private ConfigMeta getConfigMeta(Properties properties) {
        String env = getEnv(properties);
        if (ENV_LOCAL.equals(env)) {
            return null;
        }

        String key = env + "_meta";
        ConfigMeta meta = new ConfigMeta();
        meta.metaName = key;
        getNamespaces(meta, properties);
        String metaUrl = getPrioritizedProperty(properties, key);
        meta.metaUrl = metaUrl;
        return meta;
    }


    protected Properties initializeProperties(Properties sysProperties) {
        if (INITIALIZED.compareAndSet(false, true)) {
            log.info("---- start process properties ---");
            // 初始化元信息配置
            this.props = new Properties();
            String env = getEnv(sysProperties);
            setActiveProperty(ENV_KEY, env);
            String appId = getAppId();
            String cluster = getApolloCluster(sysProperties);
            setActiveProperty(APP_ID, appId);
            if (StringUtils.isNotEmpty(cluster)) {
                setActiveProperty(APOLLO_CLUSTER, cluster);
            }

            // 加载配置
            this.meta = readConfig(env, sysProperties);

            // 打印配置
            // printAllProperties();
            propConfig = this;
            matchAndProcessProps(props);
        }
        return this.props;
    }


    private void matchAndProcessProps(Properties props) {
        Enumeration enumeration = props.propertyNames();
        for (; enumeration.hasMoreElements(); ) {
            final String name = (String) enumeration.nextElement();

            for (final PropPatternProcessor patternProcessor : patternProcessorList) {
                if (patternProcessor.isMatch(name)) {
                    final Object value = props.getProperty(name);
                    patternProcessor.process(name, value);
                    addPropChangeCallback(name, new PropChangeCallback() {
                        @Override
                        public void onChange(PropChange propChange) {
                            patternProcessor.process(name, propChange.getNewValue());
                        }
                    });
                }
            }
        }
    }

    private void setActiveProperty(String name, String value) {
        if (name == null || value == null) return;
        this.props.setProperty(name, value);
        System.setProperty(name, value);
    }

    private boolean canPrintProperties() {
        String configLogEnable = System.getProperty("config.log.enable");
        if (configLogEnable == null || !"false".equals(configLogEnable.toLowerCase())) {
            if (ENV_LOCAL.equals(getEnv())) {
                return  MERGED.get() && PRINTED.compareAndSet(false, true);
            }
            return PRINTED.compareAndSet(false, true);
        }
        return false;
    }

    public void printAllProperties() {
        if (canPrintProperties()) {
            String env = getEnv();
            String appId = getAppId();
            String cluster = getApolloCluster();
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("@****************************************************************************************@\n");
            stringBuilder.append("@                               * THE BEAST CONFIGURATION *                              @\n");
            stringBuilder.append("@****************************************************************************************@\n");
            stringBuilder.append("\n");
            stringBuilder.append("------------------------------------ Service META INFO -----------------------------------\n");
            stringBuilder.append("|\n");
            stringBuilder.append("|   Environment : " + getPrintableEnvName(env) + "\n");
            stringBuilder.append("|   App ID : " + appId + "\n");
            if (StringUtils.isNotEmpty(cluster)) {
                stringBuilder.append("|   Apollo Cluster : " + cluster + "\n");
            }
            if (meta != null) {
                if (meta.metaName != null) {
                    stringBuilder.append("|   Meta Name : " + meta.metaName + "\n");
                }
                if (meta.metaUrl != null) {
                    stringBuilder.append("|   Meta URL : " + meta.metaUrl + "\n");
                }
                if (meta.namespaceList != null) {
                    stringBuilder.append("|   Namespaces : " + StringUtils.join(meta.namespaceList, ", ") + "\n");
                }
            }
            printAllProperties(meta, stringBuilder);
            stringBuilder.append("|\n");
            stringBuilder.append("@****************************************************************************************@\n");
            log.info("init properties:\n\n" + stringBuilder.toString());
        }
    }

    private void printAllProperties(ConfigMeta meta, StringBuilder stringBuilder) {
        Properties cache = new Properties();
        Iterator<String> iterator = meta.properties.keySet().iterator();
        for (; iterator.hasNext(); ) {
            String key = iterator.next();
            Properties properties = meta.properties.get(key);
            printProperties(properties, "Namespace: " + key, stringBuilder, cache);
        }
        Iterator<Map.Entry<Object, Object>> pIt = this.props.entrySet().iterator();
        Properties otherProps = new Properties();
        for (; pIt.hasNext(); ) {
            Map.Entry<Object, Object> entry = pIt.next();
            Object key = entry.getKey();
            Object value = entry.getValue();
            if (!cache.containsKey(key)) {
                otherProps.put(key, value);
            }
        }
        printProperties(otherProps, "Other Properties", stringBuilder, null);
    }

    private void printProperties(Properties properties, String title, StringBuilder stringBuilder, Properties cache) {
        if (properties.size() == 0)
            return;
        Iterator<Map.Entry<Object, Object>> it = properties.entrySet().iterator();
        stringBuilder.append("|\n");
        stringBuilder.append("------------------------------------ " + title + " -----------------------------------\n");
        stringBuilder.append("|\n");
        while (it.hasNext()) {
            Map.Entry<Object, Object> entry = it.next();
            Object key = entry.getKey();
            Object value = entry.getValue();
            if (cache != null) {
                cache.put(key, value);
            }
            stringBuilder.append("|   " + key + " = " + value + "\n");
        }
    }


    private static String getApolloEnvName(String env) {
        if (ENV_TEST.equals(env)) {
            return APOLLO_FAT;
        }
        if (ENV_PRE.equals(env)) {
            return APOLLO_UAT;
        }
        if (ENV_PROD.equals(env)) {
            return APOLLO_PRO;
        }
        return env;
    }


    private String getPrintableEnvName(String env) {
        String upEnv = env.toUpperCase();
        if (APOLLO_FAT.equals(env)) {
            return "TEST (FAT)";
        }
        if (APOLLO_UAT.equals(env)) {
            return "PRE (UAT)";
        }
        if (APOLLO_PRO.equals(env)) {
            return "PROD (PRO)";
        }
        return upEnv;
    }


    private ConfigMeta readConfig(String env, Properties properties) {
        ConfigMeta meta = null;
        if (ENV_LOCAL.equals(env)) {
            meta = readFromLocal(properties);
        }
        else {
            meta = readFromApollo(properties);
        }
        return meta;
    }

    /**
     * 从本地文件读取配置信息
     * @param properties
     */
    private ConfigMeta readFromLocal(Properties properties) {
        ConfigMeta meta = new ConfigMeta();
        meta.metaName = "local config";
        meta.namespaces = "Local Config";
        meta.namespaceList = Lists.<String>newArrayList("Local Config");
        Properties nsProps = new Properties();
        meta.properties.put(meta.namespaces, nsProps);
        for (Map.Entry<Object, Object> propIt : properties.entrySet()) {
            if (!this.props.containsKey(propIt.getKey())) {
                String key = propIt.getKey().toString();
                Object value = propIt.getValue();
                this.props.put(key, value);
                nsProps.put(key, value);
            }
        }
        return meta;
    }

    /**
     * 从Apollo读取配置信息
     * @param properties
     */
    private ConfigMeta readFromApollo(Properties properties) {
        ConfigMeta meta = getConfigMeta(properties);
        if (meta != null) {
            props.put(meta.metaName, meta.metaUrl);
            System.setProperty(meta.metaName, meta.metaUrl);
        }
        readApolloConfig("application", meta, ConfigService.getAppConfig());
        addDefaultPropertiesChangeListener();
        if (meta.namespaceList != null) {
            for (String namespace : meta.namespaceList) {
                readApolloConfig(namespace, meta, ConfigService.getConfig(namespace));
                addDefaultPropertiesChangeListener(namespace);
            }
        }
        return meta;
    }

    /**
     * 读取Apollo的Config对象中的信息
     * @param config
     */
    private void readApolloConfig(String namespace, ConfigMeta meta, Config config) {
        Properties properties = meta.properties.get(namespace);
        if (properties == null) {
            properties = new Properties();
            meta.properties.put(namespace, properties);
        }
        Set<String> names = config.getPropertyNames();
        for (String name : names) {
            if (!props.containsKey(name)) {
                String value = config.getProperty(name, null);
                namespaceMap.put(name, namespace);
                props.setProperty(name, value);
                properties.setProperty(name, value);
            }
        }
    }


    public String getProperty(String key) {
        return getPrioritizedProperty(props, key);
    }

    public String getProperty(String key, String defaultValue) {
        return props.getProperty(key, defaultValue);
    }

    public Object setProperty(String key, Object value) {
        return props.setProperty(key, value.toString());
    }

    public static Properties getPropertiesObj(){
        return getInstance().getProps();
    }



    private PropConfig addPropertiesChangeListener(String namespace, final PropChangeListener listener) {
        Config config = null;
        if (StringUtils.isBlank(namespace) || namespace.equals("application")) {
            config = ConfigService.getAppConfig();
        }
        else {
            config = ConfigService.getConfig(namespace);
        }
        config.addChangeListener(new ConfigChangeListener() {
            @Override
            public void onChange(ConfigChangeEvent changeEvent) {
                PropChangeEvent propChangeEvent = new ApolloPropChangeEvent(changeEvent);
                listener.onChange(propChangeEvent);
            }
        });
        return this;
    }


    private PropConfig addDefaultPropertiesChangeListener(String namespace) {
        log.info("%%%% 添加Apollo配置监听器 [Namespace: " + namespace  + "] %%%%");
        return addPropertiesChangeListener(namespace, new PropChangeListener() {
            @Override
            public void onChange(PropChangeEvent event) {
                List<String> propNames = event.getPropNames();
                log.info("%%% Apollo配置发生改变 %%%");
                log.info("%%% namespace: " + event.getNamespace());
                log.info("%%% propNames: " + propNames);
                String namespace = event.getNamespace();
                Map<String, Map<String, PropChangeCallback>> propChangeCallbackPropNameMap = propChangeCallbackMap.get(namespace);
                for (String propName : propNames) {
                    PropChange change = event.getChange(propName);
                    if (change != null) {
                        log.info("%%% Change Property: [Prop Name: " + change.getName()
                                + "] [New Value: " + change.getNewValue() + "] [Old Value: " + change.getOldValue() + "]");
                        props.setProperty(change.getName(), change.getNewValue());
                        Map<String, DynamicPropAccessor> propAccessorBeanMap = propAccessorsMap.get(propName);
                        if (MapUtils.isNotEmpty(propAccessorBeanMap)) {
                            for (DynamicPropAccessor propAccessor : propAccessorBeanMap.values()) {
                                propAccessor.refreshValue();
                            }
                        }
                        if (propChangeCallbackPropNameMap != null) {
                            Map<String, PropChangeCallback> callbackBeanMap = propChangeCallbackPropNameMap.get(propName);
                            if (MapUtils.isNotEmpty(callbackBeanMap)) {
                                for (PropChangeCallback callback : callbackBeanMap.values()) {
                                    callback.onChange(change);
                                }
                            }
                        }
                    }
                }
            }
        });
    }

    private PropConfig addDefaultPropertiesChangeListener() {
        return addDefaultPropertiesChangeListener("application");
    }


    public PropConfig addPropertiesChangeCallback(String namespace, String propName, PropChangeCallback callback) {
        return addPropertiesChangeCallback("default", namespace, propName, callback);
    }

    public PropConfig addPropertiesChangeCallback(String beanId, String namespace, String propName, PropChangeCallback callback) {
        if (namespace == null) {
            namespace = namespaceMap.get(propName);
            if (StringUtils.isBlank(namespace)) {
                namespace = "application";
            }
        }
        if (ENV_LOCAL.equals(props.get("env"))) {
            return this;
        }
        log.info("%%%% 添加配置监听回调方法: [Namespace: " + namespace + "] [propName: " + propName + "] %%%%");
        Map<String, Map<String, PropChangeCallback>> propChangeCallbackPropNameMap = propChangeCallbackMap.get(namespace);
        if (propChangeCallbackPropNameMap == null) {
            propChangeCallbackPropNameMap = new HashMap<>();
            propChangeCallbackMap.put(namespace, propChangeCallbackPropNameMap);
        }
        Map<String, PropChangeCallback> callbackBeanMap = propChangeCallbackPropNameMap.get(propName);
        if (callbackBeanMap == null) {
            callbackBeanMap = new LinkedHashMap<>();
            propChangeCallbackPropNameMap.put(propName, callbackBeanMap);
        }
        callbackBeanMap.put(beanId, callback);
        return this;
    }


    static void registerPropAccessor(String beanId, String propName, DynamicPropAccessor accessor) {
        Map<String, DynamicPropAccessor> propAccessorBeanMap = propAccessorsMap.get(propName);
        if (propAccessorBeanMap == null) {
            propAccessorBeanMap = new HashMap<>();
            propAccessorsMap.put(propName, propAccessorBeanMap);
        }
        propAccessorBeanMap.put(beanId, accessor);
    }




    public static void addPropChangeCallback(String beanId, String propName, PropChangeCallback callback) {
        propConfig.addPropertiesChangeCallback(beanId, null, propName, callback);
    }

    public static void addPropChangeCallback(String propName, PropChangeCallback callback) {
        propConfig.addPropertiesChangeCallback(null, propName, callback);
    }

}
