package com.thebeastshop.common.converter;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Lists;
import com.thebeastshop.common.enums.BooleanValue;
import com.thebeastshop.common.enums.CodeEnum;
import com.thebeastshop.common.exception.CodeEnumConvertException;
import com.thebeastshop.common.utils.BeanUtil;
import com.thebeastshop.common.utils.EnumUtil;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2018-04-03 13:22
 */
public class BeanFieldConverter {

    public static Object convert(Object sourceObj, Class sourceType, Class targetType) {
        return convert(sourceObj, sourceType, targetType, false);
    }

    public static Class<?> toClass(Type genericType) {
        if (genericType instanceof TypeReference) {
            return toClass(((TypeReference<?>) genericType).getType());
        }
        if (genericType instanceof ParameterizedType) {
            ParameterizedType pt = (ParameterizedType) genericType;
            return ((Class<?>) pt.getRawType());
        } else if (genericType instanceof TypeVariable) {
            TypeVariable<?> tType = (TypeVariable<?>) genericType;
            String className = tType.getGenericDeclaration().toString();
            try {
                return Class.forName(className);
            } catch (ClassNotFoundException ignored) {
            }
            return null;
        } else if (genericType instanceof WildcardType
                && "?".equals(genericType.toString())) {
            return Object.class;
        } else {
            try {
                return (Class<?>) genericType;
            } catch (Throwable th) {
                return Object.class;
            }
        }
    }


    public static Object convert(Object sourceObj, Type sourceType, Type targetType, boolean deeply) {
        if (sourceObj == null) {
            return null;
        }

        Class sourceClass = toClass(sourceType);
        Class targetClass = toClass(targetType);

        if (targetClass.isAssignableFrom(sourceClass)) {
            return sourceObj;
        }

        if (GenericBeanConverter.isIntegerNumber(targetClass)) {
            if (Short.class.isAssignableFrom(targetClass) || short.class.isAssignableFrom(targetClass)) {
                if (sourceObj instanceof Byte) {
                    return ((Byte) sourceObj).shortValue();
                }
                if (sourceObj instanceof Short) {
                    return sourceObj;
                }
                if (sourceObj instanceof Integer) {
                    return ((Integer) sourceObj).shortValue();
                }
                if (sourceObj instanceof Long) {
                    return ((Long) sourceObj).shortValue();
                }
                if (sourceObj instanceof BigDecimal) {
                    return ((BigDecimal) sourceObj).shortValue();
                }
                if (sourceObj instanceof BigInteger) {
                    return ((BigInteger) sourceObj).shortValue();
                }
            }
            if (Integer.class.isAssignableFrom(targetClass) || int.class.isAssignableFrom(targetClass)) {
                if (sourceObj instanceof Byte) {
                    return ((Byte) sourceObj).intValue();
                }
                if (sourceObj instanceof Short) {
                    return ((Short) sourceObj).intValue();
                }
                if (sourceObj instanceof Integer) {
                    return sourceObj;
                }
                if (sourceObj instanceof Long) {
                    return ((Long) sourceObj).intValue();
                }
                if (sourceObj instanceof BigDecimal) {
                    return ((BigDecimal) sourceObj).intValue();
                }
                if (sourceObj instanceof BigInteger) {
                    return ((BigInteger) sourceObj).intValue();
                }
            }
            if (Long.class.isAssignableFrom(targetClass) || long.class.isAssignableFrom(targetClass)) {
                if (sourceObj instanceof Byte) {
                    return ((Byte) sourceObj).longValue();
                }
                if (sourceObj instanceof Short) {
                    return ((Short) sourceObj).longValue();
                }
                if (sourceObj instanceof Integer) {
                    return ((Integer) sourceObj).longValue();
                }
                if (sourceObj instanceof Long) {
                    return sourceObj;
                }
                if (sourceObj instanceof BigDecimal) {
                    return ((BigDecimal) sourceObj).longValue();
                }
                if (sourceObj instanceof BigInteger) {
                    return ((BigInteger) sourceObj).longValue();
                }
            }
            if (BigInteger.class.isAssignableFrom(targetClass)
                    && GenericBeanConverter.isIntegerNumber(sourceClass)) {
                return new BigInteger(sourceObj + "");
            }
            if (BigDecimal.class.isAssignableFrom(targetClass)
                    && (GenericBeanConverter.isIntegerNumber(sourceClass)
                    || GenericBeanConverter.isDecimalNumber(sourceClass))) {
                return new BigDecimal(sourceObj + "");
            }
            if (GenericBeanConverter.isPrimaryIntegerNumber(sourceClass)) {
                return targetClass.cast(sourceObj);
            }
        }
        if (GenericBeanConverter.isPrimaryDecimalNumber(targetClass)) {
            if (GenericBeanConverter.isPrimaryDecimalNumber(sourceClass)) {
                return targetClass.cast(sourceObj);
            }
            if (Float.class.isAssignableFrom(targetClass) || float.class.isAssignableFrom(targetClass)) {
                if (sourceObj instanceof BigDecimal) {
                    return ((BigDecimal) sourceObj).floatValue();
                }
            }
            if (Double.class.isAssignableFrom(targetClass) || double.class.isAssignableFrom(targetClass)) {
                if (sourceObj instanceof BigDecimal) {
                    return ((BigDecimal) sourceObj).doubleValue();
                }
            }
        }

        if (Integer.class.isAssignableFrom(sourceClass) && Boolean.class.isAssignableFrom(targetClass)) {
            if ((Integer)sourceObj == 0) {
                return false;
            }
            return true;
        }
        if (Boolean.class.isAssignableFrom(sourceClass) && Integer.class.isAssignableFrom(targetClass)) {
            if ((Boolean)sourceObj == false) {
                return 0;
            }
            return 1;
        }
        if (sourceObj instanceof BooleanValue) {
            if (targetClass.isAssignableFrom(Boolean.class) || targetClass.isAssignableFrom(boolean.class)) {
                BooleanValue booleanValue = (BooleanValue) sourceObj;
                return booleanValue.toBooleanValue();
            }
        }
        if (sourceObj instanceof CodeEnum) {
            CodeEnum codeEnum = (CodeEnum)sourceObj;
            Object code = codeEnum.getCode();
            Class codeType = code.getClass();
            if (!targetClass.isAssignableFrom(codeType)) {
                throw new CodeEnumConvertException(
                        sourceClass.getName() +
                        "没有包含正确的泛型类型参数，CodeEnum<>泛型参数中的" +
                        codeType.getName() + "不能转换为" + targetClass.getName());
            }
            return code;
        }
        if (Enum.class.isAssignableFrom(targetClass)) {
            if (BooleanValue.class.isAssignableFrom(targetClass)) {
                if (Boolean.class.isAssignableFrom(sourceClass) || boolean.class.isAssignableFrom(sourceClass)) {
                    return EnumUtil.getByBooleanValue(targetClass, (Boolean) sourceObj);
                }
            }
            if (CodeEnum.class.isAssignableFrom(targetClass)) {
                Type[] interfaces = targetClass.getGenericInterfaces();
                for (Type type : interfaces) {
                    if (type instanceof ParameterizedType) {
                        ParameterizedType parameterizedType = (ParameterizedType) type;
                        Class rawType = (Class) parameterizedType.getRawType();
                        if (CodeEnum.class.isAssignableFrom(rawType)) {
                            Type[] argTypes = parameterizedType.getActualTypeArguments();
                            if (argTypes.length != 1) {
                                throw new CodeEnumConvertException(
                                        targetClass.getName() +
                                                "没有包含正确的泛型类型参数");
                            }
                            Class argType = (Class) argTypes[0];
                            if (!argType.isAssignableFrom(sourceClass)) {
                                throw new CodeEnumConvertException(
                                        targetClass.getName() +
                                                "没有包含正确的泛型类型参数， " +
                                                sourceClass.getName() +
                                                "无法转换为CodeEnum<>泛型参数中的" + argType.getName());
                            }
                            return EnumUtil.getByCode(targetClass, sourceObj);
                        }
                    }
                }
            }
        }
        if (deeply) {
            if (targetType instanceof ParameterizedType && sourceType instanceof ParameterizedType) {
                ParameterizedType parameterizedTargetType = (ParameterizedType) targetType;
                ParameterizedType parameterizedSourceType = (ParameterizedType) sourceType;
                Class targetRawClass = (Class) parameterizedTargetType.getRawType();
                Class sourceRawClass = (Class) parameterizedSourceType.getRawType();
                if (Serializable.class.isAssignableFrom(targetRawClass) && Serializable.class.isAssignableFrom(sourceRawClass)) {
                    if (Collection.class.isAssignableFrom(sourceClass) && Collection.class.isAssignableFrom(targetClass)) {
                        return BeanUtil.buildListFrom((Collection) sourceObj, targetRawClass, deeply);
                    }
                    if (Map.class.isAssignableFrom(sourceClass) && Map.class.isAssignableFrom(targetClass)) {
                        return BeanUtil.buildMapFrom((Map) sourceObj, targetRawClass, deeply);
                    }
                }
            }

            if (Serializable.class.isAssignableFrom(sourceClass) && Serializable.class.isAssignableFrom(targetClass)) {
                return BeanUtil.buildFrom(sourceObj, targetClass, true);
            }
        }
        return null;
    }

    public static class SrcGood implements Serializable {
        private String name;
        private float price;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public float getPrice() {
            return price;
        }

        public void setPrice(float price) {
            this.price = price;
        }
    }

    public static class SrcUser implements Serializable {

        private String username;

        private String password;

        private List<SrcGood> goods;

        public String getUsername() {
            return username;
        }

        public void setUsername(String username) {
            this.username = username;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public List<SrcGood> getGoods() {
            return goods;
        }

        public void setGoods(List<SrcGood> goods) {
            this.goods = goods;
        }
    }


    public static class TargetGood implements Serializable {
        private String name;
        private float price;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public float getPrice() {
            return price;
        }

        public void setPrice(float price) {
            this.price = price;
        }
    }


    public static class TargetUser implements Serializable {

        private String username;

        private String password;

        private List<TargetGood> goods;

        public String getUsername() {
            return username;
        }

        public void setUsername(String username) {
            this.username = username;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public List<TargetGood> getGoods() {
            return goods;
        }

        public void setGoods(List<TargetGood> goods) {
            this.goods = goods;
        }
    }



    public static void main(String[] args) {
        int iiret = (int) BeanFieldConverter.convert(123, int.class, int.class);
        System.out.println("int to int -> " + iiret);

        int integeriret = (int) BeanFieldConverter.convert(123, Integer.class, int.class);
        System.out.println("Integer to int -> " + integeriret);

        Integer iintegerret = (Integer) BeanFieldConverter.convert(123, int.class, Integer.class);
        System.out.println("int to Integer -> " + iintegerret);

        BigDecimal idret = (BigDecimal) BeanFieldConverter.convert(123, Integer.class, BigDecimal.class);
        System.out.println("int to big decimal -> " + idret);

        BigDecimal ldret = (BigDecimal) BeanFieldConverter.convert(1230000L, Long.class, BigDecimal.class);
        System.out.println("long to big decimal -> " + ldret);

        Integer diret = (Integer) BeanFieldConverter.convert(new BigDecimal("234"), BigDecimal.class, Integer.class);
        System.out.println("big decimal to int -> " + diret);

        Long dlret = (Long) BeanFieldConverter.convert(new BigDecimal("2340000"), BigDecimal.class, Long.class);
        System.out.println("big decimal to long -> " + dlret);

        Long ilret = (Long) BeanFieldConverter.convert(200, Integer.class, Long.class);
        System.out.println("int to long -> " + ilret);

        Integer liret = (Integer) BeanFieldConverter.convert(Long.valueOf(20000L), Long.class, Integer.class);
        System.out.println("long to int -> " + liret);

        SrcUser srcUser1 = new SrcUser();
        srcUser1.setUsername("xxx");
        srcUser1.setPassword("yyy");
        SrcGood good1 = new SrcGood();
        good1.setName("Good1");
        good1.setPrice(14.5f);
        SrcGood good2 = new SrcGood();
        good2.setName("Good1");
        good2.setPrice(14.5f);
        srcUser1.setGoods(Lists.newArrayList(good1, good2));

        SrcUser srcUser2 = new SrcUser();
        srcUser2.setUsername("foo");
        srcUser2.setPassword("bar");

        List<SrcUser> users = Lists.newArrayList(srcUser1, srcUser2);

        List<TargetUser> targetUsers = BeanUtil.buildListFrom(users, TargetUser.class, false);
        System.out.println(JSON.toJSONString(targetUsers));


    }

}
