package com.thebeastshop.common.converter;

import com.thebeastshop.common.enums.BooleanValue;
import com.thebeastshop.common.enums.CodeEnum;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.FatalBeanException;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

import java.beans.PropertyDescriptor;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


/**
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-04-25 17:11
 */
public class GenericBeanConverter implements BeanConverter {

    public <T> ToList<T> from(Class<T> source) {
        return ConverterSupport.from(source);
    }

    protected static BeanConverterHandlerManager matchHandlerManager(Class<?> fromClass, Class<?> toClass) {
        return ConverterSupport.matchHandlerManager(fromClass, toClass);
    }


    @Override
    public <T> T converterFrom(Object obj, Class<T> clazz) {
        if (obj == null) {
            return null;
        }
        T ret = null;
        try {
            ret = clazz.newInstance();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        if (ret == null) {
            return null;
        }
        Class fromClass = obj.getClass();
        BeanConverterHandlerManager handlerManager = matchHandlerManager(fromClass, clazz);
        if (handlerManager != null) {
            String[] ignoreFields = handlerManager.getIgnoreFields();
            copyProperties(obj, ret, ignoreFields);
            handlerManager.afterCopyProperties(obj, ret);
        }
        else {
            BeanUtils.copyProperties(obj, ret);
        }
        return ret;
    }

    public static void copyProperties(Object source, Object target, String... ignoreProperties) throws BeansException {
        copyProperties(source, target, null, ignoreProperties);
    }

    public static void copyProperties(Object source, Object target, Class<?> editable, String... ignoreProperties)
            throws BeansException {
        copyProperties(source, target, editable, false, ignoreProperties);
    }

    public static void copyProperties(Object source, Object target, Class<?> editable, boolean deeply, String... ignoreProperties)
            throws BeansException {

        Assert.notNull(source, "Source must not be null");
        Assert.notNull(target, "Target must not be null");

        Class<?> actualEditable = target.getClass();
        if (editable != null) {
            if (!editable.isInstance(target)) {
                throw new IllegalArgumentException("Target class [" + target.getClass().getName() +
                        "] not assignable to Editable class [" + editable.getName() + "]");
            }
            actualEditable = editable;
        }
        PropertyDescriptor[] targetPds = BeanUtils.getPropertyDescriptors(actualEditable);
        List<String> ignoreList = (ignoreProperties != null ? Arrays.asList(ignoreProperties) : null);

        for (PropertyDescriptor targetPd : targetPds) {
            Method writeMethod = targetPd.getWriteMethod();
            if (writeMethod != null && (ignoreList == null || !ignoreList.contains(targetPd.getName()))) {
                PropertyDescriptor sourcePd = BeanUtils.getPropertyDescriptor(source.getClass(), targetPd.getName());
                if (sourcePd != null) {
                    Method readMethod = sourcePd.getReadMethod();
                    if (readMethod != null) {
                        Class writeType = writeMethod.getParameterTypes()[0];
                        Class readType = readMethod.getReturnType();
                        Type writeGenericType = writeMethod.getGenericParameterTypes()[0];
                        Type readGenericType =  readMethod.getGenericReturnType();
                        if (checkType(writeType, readType, deeply)) {
                            try {
                                if (!Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
                                    readMethod.setAccessible(true);
                                }
                                Object value = readMethod.invoke(source);
                                if (!Modifier.isPublic(writeMethod.getDeclaringClass().getModifiers())) {
                                    writeMethod.setAccessible(true);
                                }
                                Object convertedValue = BeanFieldConverter.convert(value, readGenericType, writeGenericType, deeply);
                                if (convertedValue != null) {
                                    writeMethod.invoke(target, convertedValue);
                                }
                            } catch (Throwable ex) {
                                throw new FatalBeanException(
                                        "Could not copy property '" + targetPd.getName() + "' from source to target", ex);
                            }
                        }
                    }
                }
            }
        }
    }


    public static boolean isPrimaryIntegerNumber(Class type) {
        return Short.class.isAssignableFrom(type)
                || short.class.isAssignableFrom(type)
                || Integer.class.isAssignableFrom(type)
                || int.class.isAssignableFrom(type)
                || Long.class.isAssignableFrom(type)
                || long.class.isAssignableFrom(type);
    }

    public static boolean isIntegerNumber(Class type) {
        return isPrimaryIntegerNumber(type)
                || BigInteger.class.isAssignableFrom(type)
                || BigDecimal.class.isAssignableFrom(type);
    }

    public static boolean isPrimaryDecimalNumber(Class type) {
        return Float.class.isAssignableFrom(type)
                || Double.class.isAssignableFrom(type);
    }

    public static boolean isDecimalNumber(Class type) {
        return isPrimaryDecimalNumber(type)
                || BigDecimal.class.isAssignableFrom(type);
    }

    private static boolean checkType(Class writeType, Class readType) {
        return checkType(writeType, readType, false);
    }

    private static boolean checkType(Class writeType, Class readType, boolean deeply) {
        if (ClassUtils.isAssignable(writeType, readType)) {
            return true;
        }
        if (Integer.class.isAssignableFrom(writeType) && Boolean.class.isAssignableFrom(readType)) {
            return true;
        }
        if (Integer.class.isAssignableFrom(readType) && Boolean.class.isAssignableFrom(writeType)) {
            return true;
        }
        if (isIntegerNumber(writeType) && isIntegerNumber(readType)) {
            return true;
        }
        if (isIntegerNumber(readType) && isIntegerNumber(writeType)) {
            return true;
        }
        if (BooleanValue.class.isAssignableFrom(readType) || BooleanValue.class.isAssignableFrom(writeType)) {
            return true;
        }
        if (CodeEnum.class.isAssignableFrom(readType) || CodeEnum.class.isAssignableFrom(writeType)) {
            return true;
        }
        if (Serializable.class.isAssignableFrom(readType) && Serializable.class.isAssignableFrom(writeType)) {
            return true;
        }
        return false;
    }


    @Override
    public <T> List<T> converterListFrom(List list, Class<T> clazz) {
        List<T> results = new ArrayList<>();
        for (Object obj : list) {
            results.add(converterFrom(obj, clazz));
        }
        return results;
    }

    public GenericBeanConverter() {
        init();
    }

    protected void init() {
    }
}
