package com.thebeastshop.common.utils;

import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 日期工具类
 *
 * @author bryan.zhang
 */
public class DateUtil {

    private static Logger logger = LoggerFactory.getLogger(DateUtil.class);

    /** yyyy-MM-dd */
    public static final String FORMAT_YYYY_MM_DD = "yyyy-MM-dd";

    /** yyyyMMdd */
    public static final String FORMAT_YYYYMMDD = "yyyyMMdd";

    /** yyMMdd */
    public static final String FORMAT_YYMMDD = "yyMMdd";

    /** yyyy-MM-dd HH:mm:ss */
    public static final String FORMAT_YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    /** yyyyMMddHHmmss */
    public static final String FORMAT_YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

    /** yyyyMMddHHmm */
    public static final String FORMAT_YYYYMMDDHHMM = "yyyyMMddHHmm";

    /** HH:mm:ss */
    public static final String FORMAT_HH_MM_SS = "HH:mm:ss";

    /** HHmmss */
    public static final String FORMAT_HHMMSS = "HHmmss";

    /** dd */
    public static final String FORMAT_DD = "dd";

    /** MMdd */
    public static final String FORMAT_MMDD = "MMdd";

    public static final String FORMAT_YYYY_MM_DD_HH_MM = "yyyy-MM-dd HH:mm";

    public static final String FORMAT_UTC = "yyyy-MM-dd'T'HH:mm:ss'Z'";

    public static final String FORMAT_MICROSECOND = "yyyyMMddHHmmssmmm";
    public static final String FORMAT_DATE_2 = "yyyyMMdd";
    public static final String FORMAT_DATE_3 = "MM/dd/yyyy";
    public static final String FORMAT_DATE_4 = "MM/dd/yyyy HH:mm";
    public static final String FORMAT_MM_DD = "MM-dd";
    public static final String FORMAT_M_YUE_D_RI = "M月d日";
    public static final String MONGOSTRINGTODATE = "EEE MMM dd HH:mm:ss zzz yyyy";

    /**
     * 将日期转换成字符格式
     *
     * @param date
     *            java.util.Date类型
     * @param format
     *            如果为null或""，默认为DATE格式
     * @return 无法成功转换则返回null
     */
    public static String date2String(Date date, String format) {
        String result = null;
        if (date == null) {
            return result;
        }
        if (StringUtils.isEmpty(format)) {
            format = FORMAT_YYYY_MM_DD;
        }
        try {
            result = formatDate(date, format);
        } catch (Exception ex) {
            logger.warn("日期转换为字符串错误，日期：" + date.toString() + "， 格式：" + format);
        }

        return result;
    }

    public static String workDays(String strStartDate, String strEndDate) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        Calendar cl1 = Calendar.getInstance();
        Calendar cl2 = Calendar.getInstance();

        try {
            cl1.setTime(df.parse(strStartDate));
            cl2.setTime(df.parse(strEndDate));

        } catch (ParseException e) {
            logger.warn("日期格式非法");
            e.printStackTrace();
        }

        Long time =  cl2.getTimeInMillis() - cl1.getTimeInMillis();

        int count = 0;
        while (cl1.compareTo(cl2) <= 0) {
            if (cl1.get(Calendar.DAY_OF_WEEK) != 7 && cl1.get(Calendar.DAY_OF_WEEK) != 1)
                count++;
            cl1.add(Calendar.DAY_OF_MONTH, 1);
        }

        int nDays =  nDaysBetweenTwoDate(strStartDate, strEndDate);
        int minute = (int) (time/(1000 * 60));

        int differenceHours = (minute - 60 * 24 * nDays)/60;

        return count - 1 + "天" + Math.abs(differenceHours) + "小时";

    }


    // 计算两个日期相隔的天数
    public static int nDaysBetweenTwoDate(String firstString, String secondString) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date firstDate = null;
        Date secondDate = null;
        try {
            firstDate = df.parse(firstString);
            secondDate = df.parse(secondString);
        } catch (Exception e) {
            // 日期型字符串格式错误
            logger.warn("日期型字符串格式错误");
        }
        int nDay = (int) ((secondDate.getTime() - firstDate.getTime()) / (24 * 60 * 60 * 1000));
        return nDay;
    }



    /**
     * 字符串 转 日期
     *
     * @param dateString
     *            日期字符串
     * @return 日期对象
     */
    public static final Date parseDate_YYYYMMDD(String dateString) {
        return parseDate(dateString, FORMAT_YYYYMMDD);
    }

    public static final Date parseDate_YYYYMMDDHHMM(String dateString) {
        return parseDate(dateString, FORMAT_YYYYMMDDHHMM);
    }

    public static final Date parseDate_YYYYMMDDHHMMSS(String dateString) {
        return parseDate(dateString, FORMAT_YYYYMMDDHHMMSS);
    }

    public static final Date parseDate(String dateString, String pattern) {
        Date date = null;
        if (StringUtils.isEmpty(dateString))
            return null;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            date = sdf.parse(dateString);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return date;
    }


    /**
     * 日期对象格式化
     *
     * @param date
     *            日期对象
     * @param pattern
     *            格式化样式
     * @return 字符串日期
     */
    public static final String formatDate(Date date, String pattern) {
        String v = null;
        try {
            if (date == null)
                return null;
            SimpleDateFormat dateFormat = new SimpleDateFormat(pattern);
            v = dateFormat.format(date);
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    /**
     * 日期对象格式化
     *
     * @param date
     *            日期对象
     * @return 字符串日期
     */
    public static final String formatDate_YYYY_MM_DD(Date date) {
        String v = null;
        try {
            if (date == null)
                return null;
            SimpleDateFormat dateFormat = new SimpleDateFormat(FORMAT_YYYY_MM_DD);
            v = dateFormat.format(date);
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    public static final String formatDate_YYYYMMDD(Date date) {
        String v = null;
        try {
            if (date == null)
                return null;
            SimpleDateFormat dateFormat = new SimpleDateFormat(FORMAT_YYYYMMDD);
            v = dateFormat.format(date);
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    /**
     * 判断传入的日期是否是今天
     * <p>
     * 2002-03-28 13:45 和 2002-03-28 返回 true. 2002-03-12 13:45 和 2002-03-28 返回
     * false.
     * </p>
     *
     * @param date
     *            需要判断的日期对象
     * @return 是今天返回true,否则返回false
     */
    public static boolean isToday(Date date) {
        if (date == null)
            return false;
        return DateUtils.isSameDay(date, new Date());
    }


    /**
     * 判断时间是否早于当前时间
     * <p>
     * 例如:今天是2002-03-28 13:45:00, 传入2002-03-27 11:00:00 返回true,传入2002-03-28
     * 11:00:00返回false
     * </p>
     *
     * @param date
     *            需要判断的时间对象
     * @return 如果传入的时间对象小于今天(时分秒都为0),则返回true,否则返回false
     */
    public static boolean isBeforeToday(Date date) {
        if (date == null)
            return false;
        long today = DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH).getTime();
        return date.getTime() < today;
    }


    /**
     * 判断时间是否早于当前时间
     * <p>
     * 例如:今天是2002-03-28 13:45:00, 传入2002-03-27 11:00:00 返回true,传入2002-03-28
     * 11:00:00返回false
     * </p>
     *
     * @param dateString
     *            日期字符串
     * @param sFormat
     *            日期字符串的原始格式,例如yyyy-MM-dd
     * @return
     */
    public static boolean isBeforeToday(String dateString, String sFormat) {
        return isBeforeToday(parseDate(dateString, sFormat));
    }


    /**
     * 日期字符串之间的类型转换.
     * <p>
     * 例如:convertDate("2012-01-02", "yyyy-MM-dd", "yyyy/mm/dd")返回2012/01/02
     * </p>
     *
     * @param source
     *            待处理的日期字符串
     * @param sformat
     *            原来的格式
     * @param dformat
     *            新的格式
     * @return 转换后的日期字符串
     */
    public static String convertDate(String source, String sformat, String dformat) {
        // 参数检查
        if (StringUtils.isEmpty(source) || StringUtils.isEmpty(sformat) || StringUtils.isEmpty(dformat))
            return source;
        // 开始转换
        String newString = formatDate(parseDate(source, sformat), dformat);
        // 如果转换失败返回原始字符串
        return (newString == null) ? source : newString;
    }


    /**
     * 获得当前日期字符串,格式为 yyyyMMdd
     *
     * @return 当前日期字符串(yyyyMMdd)
     */
    public static String getCurrDate_HHMMSS() {
        return formatDate(new Date(), FORMAT_HHMMSS);
    }


    public static String getDate_DD(Date date) {
        return formatDate(date, FORMAT_DD);
    }


    /**
     * 获得当前日期字符串,格式为 yyyyMMdd
     *
     * @return 当前日期字符串(yyyyMMdd)
     */
    public static String getCurrDate_YYYYMMDD() {
        return formatDate(new Date(), FORMAT_YYYYMMDD);
    }


    /**
     * 获得当前日期字符串,格式为 yyyy-MM-dd
     *
     * @return 当前日期字符串(yyyy-MM-dd)
     */
    public static String getCurrDate_YYYY_MM_DD() {
        return formatDate(new Date(), FORMAT_YYYY_MM_DD);
    }


    /**
     * 获得当前日期字符串,格式为 yyyy-MM-dd HH:mm:ss
     *
     * @return 当前日期字符串(yyyy-MM-dd HH:mm:ss)
     */
    public static String getCurrDate_YYYY_MM_DD_HH_MM_SS() {
        return formatDate(new Date(), FORMAT_YYYY_MM_DD_HH_MM_SS);
    }


    /**
     * 获得当前日期字符串,格式为 yyyyMMddHHmmss
     *
     * @return 当前日期字符串(yyyyMMddHHmmss)
     */
    public static String getCurrTime_YYYYMMDDHHMMSS() {
        return formatDate(new Date(), FORMAT_YYYYMMDDHHMMSS);
    }


    public static int getYesterDay() {
        Date yesterday = calDate(new Date(), 0, 0, -1);
        return Integer.parseInt(getDate_DD(yesterday));
    }


    public static String getYesterDay_MMDD() {
        Date yesterday = calDate(new Date(), 0, 0, -1);
        return formatDate(yesterday, FORMAT_MMDD);
    }


    public static String getYesterDay_YYYYMMDD() {
        Date yesterday = calDate(new Date(), 0, 0, -1);
        return formatDate(yesterday, FORMAT_YYYYMMDD);
    }


    public static Date getLastMonthDate(int day) {
        Date lastMonth = calDate(new Date(), 0, -1, 0);
        Calendar cal = Calendar.getInstance();
        cal.setTime(lastMonth);
        if (day == 31) {
            cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
        } else {
            cal.set(Calendar.DATE, day);
        }
        return cal.getTime();
    }


    public static Date getCurMonthDate(int day) {
        Date cur = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(cur);
        if (day == 31) {
            cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
        } else {
            cal.set(Calendar.DATE, day);
        }
        return cal.getTime();
    }


    /**
     * 判断是否是日期
     *
     * @return dateString 日期字符串
     * @param pattern
     *            格式化样式(yyyyMMddHHmmss)
     * @return 是日期返回true, 否则返回false
     */
    public static boolean isDateString(String dateString, String pattern) {
        boolean v = false;
        try {
            if (StringUtils.isNotEmpty(dateString)) {
                SimpleDateFormat dateFormat = new SimpleDateFormat(pattern);
                Date d = dateFormat.parse(dateString);
                if (d != null) {
                    v = true;
                }
            }
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    /**
     * 判断是否是日期
     *
     * @return dateString 日期字符串
     * @param pattern
     *            格式化样式(yyyyMMddHHmmss)
     * @return 不是日期返回true, 否则返回false
     */
    public static boolean isNotDateString(String dateString, String pattern) {
        return !isDateString(dateString, pattern);
    }


    /**
     * 计算日期
     *
     * @param date
     * @param yearNum
     * @param monthNum
     * @param dateNum
     * @return
     */
    public static String calDate(String date, int yearNum, int monthNum, int dateNum) {
        String result = "";
        try {
            SimpleDateFormat sd = new SimpleDateFormat(FORMAT_YYYYMMDD);
            Calendar cal = Calendar.getInstance();
            cal.setTime(sd.parse(date));
            cal.add(Calendar.MONTH, monthNum);
            cal.add(Calendar.YEAR, yearNum);
            cal.add(Calendar.DATE, dateNum);
            result = sd.format(cal.getTime());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }


    public static Date calDate(Date date, int yearNum, int monthNum, int dateNum) {
        Date result = null;
        try {
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            cal.add(Calendar.MONTH, monthNum);
            cal.add(Calendar.YEAR, yearNum);
            cal.add(Calendar.DATE, dateNum);
            result = cal.getTime();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    public static Date addDate(Date date,int dateNum){
        return calDate(date, 0, 0, dateNum);
    }

    public static Date addMin(Date date, int min) {
        Date result = null;
        try {
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            cal.add(Calendar.MINUTE,min);
            result = cal.getTime();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    public static Date getDate(long time){
        return new Date(time);
    }

    /**
     * 判断是否是明天
     *
     * @param date
     * @return
     */
    public static boolean isTommorow(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.set(Calendar.DATE, c.get(Calendar.DATE) + 1);
        Date yesterday = c.getTime();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        return format.format(yesterday).equals(format.format(date));
    }

    /**
     * 判断是否是昨天
     *
     * @param date
     * @return
     */
    public static boolean isYesterday(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.set(Calendar.DATE, c.get(Calendar.DATE) - 1);
        Date yesterday = c.getTime();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        return format.format(yesterday).equals(format.format(date));
    }

    public static Date timeStamp2Date(Integer timestampString){
        Long timestamp = Long.parseLong(timestampString.toString())*1000;
        Date date = new Date(timestamp);
        return date;
    }

    public static boolean isOverlap(Date leftStartDate, Date leftEndDate,Date rightStartDate, Date rightEndDate) {
        return
                ((leftStartDate.getTime() >= rightStartDate.getTime())
                        && leftStartDate.getTime() < rightEndDate.getTime())
                        ||
                        ((leftStartDate.getTime() > rightStartDate.getTime())
                                && leftStartDate.getTime() <= rightEndDate.getTime())
                        ||
                        ((rightStartDate.getTime() >= leftStartDate.getTime())
                                && rightStartDate.getTime() < leftEndDate.getTime())
                        ||
                        ((rightStartDate.getTime() > leftStartDate.getTime())
                                && rightStartDate.getTime() <= leftEndDate.getTime());
    }

    public static Date parseUTCDate(String date, String pattern){
        try{
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setTimeZone(TimeZone.getTimeZone("UTC"));
            Date resultDate = sdf.parse(date);
            return resultDate;
        }catch (Throwable t){
            //do nothing
        }
        return null;
    }

    public static Date parseUTCDate(String date){
        return parseUTCDate(date,FORMAT_UTC);
    }

    public static final String  DEFAULT_DATE_FORMAT     = "yyyy-MM-dd";

    public static final String  DEFAULTDATEFORMAT     = "yyyyMMdd";

    public static final String  YEAR_FORMAT = "yyyy";

    public static final String  YEAR_MONTH_BAR_FORMAT     = "yyyy-MM";

    public static final String  YEAR_MONTH_FORMAT     = "yyyy年MM月";

    public static final String  DEFAULT_CN_DATE_FORMAT     = "yyyy年MM月dd日";

    public static final String  DEFAULT_TIME_FORMAT     = "HH:mm:ss";

    public static final String  DATE_STR_FORMAT_SSS     = "HH:mm:ss:SSS";

    public static final String  DEFAULT_DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

    public static final String  DEFAULT_STR_FORMAT      = "yyyyMMddHHmmss";

    public static final String  DATE_STR_FORMAT      = "yyyyMMddHHmmssSSS";

    public static final String  DATE_FORMAT            = "dd";

    public static final String  MOUNTH_FORMAT          = "MM";

    public static final Integer  WORK_DAY          = 0;
    public static final Integer  WEEK_DAY          = 1;
    public static final Integer  HOLIDAY          = 2;


    private final static SimpleDateFormat getFormat(String pattern, Locale locale) {
        return new SimpleDateFormat(pattern, locale);
    }

    /**
     * format 日期类型 格式化成字符串类型
     *
     * @param date
     * @param pattern
     * @return
     */
    public static String format(Date date, String pattern) {
        return format(date, pattern, Locale.getDefault());
    }

    /**
     * format 日期类型 格式化成字符串类型
     *
     * @param date
     * @param pattern
     * @param locale
     * @return
     */
    public static String format(Date date, String pattern, Locale locale) {
        if(date==null){
            return "";
        }
        if (org.apache.commons.lang.StringUtils.isNotBlank(pattern)) {
            String format = getFormat(pattern, locale).format(date);
            return format;
        }
        throw new IllegalArgumentException("param pattern can not be null");
    }

    /**
     * parse 字符串类型转成日期类型
     *
     * @param dateString
     * @param pattern
     * @return
     */
    public static Date parse(String dateString, String pattern) {
        return parse(dateString, pattern, Locale.getDefault());
    }

    /**
     * 字符串类型转成日期类型
     *
     * @param dateString
     * @param pattern
     * @param locale
     * @return
     */
    public static Date parse(String dateString, String pattern, Locale locale) {
        if (org.apache.commons.lang.StringUtils.isNotBlank(dateString)) {
            ParsePosition parsePosition = new ParsePosition(0);
            Date date = getFormat(pattern, locale).parse(dateString, parsePosition);
            Object[] objects = { dateString, pattern };
            return date;
        }
        throw new IllegalArgumentException("param dateString can not be null");
    }

    /**
     * 增加或减少指定数量的时间
     *
     * @param date   时间
     * @param field  计算域
     * @param amount 数值
     */
    public static Date add(Date date, int field, int amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(field, amount);
        return calendar.getTime();
    }

    /**
     * 增加或减少指定天数
     *
     * @param date
     * @param amount
     * @return
     */
    public static Date addDay(Date date, int amount) {
        return add(date, Calendar.DAY_OF_YEAR, amount);
    }

    public static Date addBeforeMonthBeginDay(Date date){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        String strY = null;
        int x = calendar.get(Calendar.YEAR);
        int y = calendar.get(Calendar.MONTH);
        strY = y >= 10 ? String.valueOf(y) : ("0" + y);
        String beginDay = x + "-" + strY + "-01";
        return parse(beginDay,"yyyy-MM-dd");
    }

    public static Date addBeforeMonthEndDay(Date date){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        String strY = null;
        String strZ = null;
        boolean leap = false;
        int x = calendar.get(Calendar.YEAR);
        int y = calendar.get(Calendar.MONTH);
        if (y == 1 || y == 3 || y == 5 || y == 7 || y == 8 || y == 10 || y == 12) {
            strZ = "31";
        }
        if (y == 4 || y == 6 || y == 9 || y == 11) {
            strZ = "30";
        }
        if (y == 2) {
            leap = leapYear(x);
            if (leap) {
                strZ = "29";
            }else {
                strZ = "28";
            }
        }
        strY = y >= 10 ? String.valueOf(y) : ("0" + y);
        String endDay =  x + "-" + strY + "-" + strZ;
        return parse(endDay,"yyyy-MM-dd");
    }

    /**
     * 功能：判断输入年份是否为闰年<br>
     *
     * @param year
     * @return 是：true  否：false
     * @author pure
     */
    public static boolean leapYear(int year) {
        boolean leap;
        if (year % 4 == 0) {
            if (year % 100 == 0) {
                if (year % 400 == 0) leap = true;
                else leap = false;
            }
            else leap = true;
        }
        else leap = false;
        return leap;
    }

    /**
     * 添加年。
     *
     * @param date
     *            日期
     * @param num
     *            添加的年数
     * @return 添加后的日期
     */
    public static Date addYears(Date date, int num) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.YEAR, num);
        return cal.getTime();
    }
    /**
     * 增加或减少指定分钟
     *
     * @param date
     * @param amount
     * @return
     */
    public static Date addMinutes(Date date, int amount) {
        return add(date, Calendar.MINUTE, amount);
    }

    /**
     * 获得指定时间当天 00:00:00 的Date对象
     *
     * @param date 指定时间
     * @return 结果
     */
    public static Date dayStart(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND,0);
        return calendar.getTime();
    }

    public static Date yesterdayStart(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, -1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND,0);
        return calendar.getTime();
    }

    /**
     * 获得指定时间当天 23:59:59 的Date对象
     *
     * @param date 指定时间
     * @return 结果
     */
    public static Date dayEnd(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND,999);
        return calendar.getTime();
    }

    public static Date dayEndSecond(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND,000);
        return calendar.getTime();
    }

    public static Date yesterdayEnd(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, -1);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND,999);
        return calendar.getTime();
    }

    /**
     * 获得中文周N
     *
     * @return 结果
     */
    public static String weekDayInChinese() {
        return weekDayInChinese(new Date());
    }

    /**
     * 根据指定日期获得中文周N
     *
     * @param date 指定日期
     * @return 结果
     */
    public static String weekDayInChinese(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int week = calendar.get(GregorianCalendar.DAY_OF_WEEK);
        switch (week) {
            case 1:
                return "日";
            case 2:
                return "一";
            case 3:
                return "二";
            case 4:
                return "三";
            case 5:
                return "四";
            case 6:
                return "五";
            case 7:
                return "六";
            default:
                return "";
        }
    }

    /**
     *  获取当前系统时间
     * @return
     */
    public static Date getNow() {
        return new Date();
    }

    public static String getNowFormat(String foramt){
        return format(getNow(),foramt);
    }

    public static Date getCurrDateByDateStr(String dateStr) {
        Date date = null;
        try {
            date = new SimpleDateFormat(DEFAULT_DATE_FORMAT).parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return date;
    }

    public static int getCurrYear() {
        return Calendar.getInstance().get(Calendar.YEAR);
    }

    public static int getCurrMonth() {
        return Calendar.getInstance().get(Calendar.MONTH)+1;
    }

    public static Date getPreMonth() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH,-1);
        return calendar.getTime();
    }

    public static Date getYesterday() {
        return getPreDay(-1);
    }

    public static Date getPreDay(int preDay) {
        Calendar calendar = Calendar.getInstance();
        return addDay(calendar.getTime(),preDay);
    }

    public static Date getCurrMonthDate() {
        Calendar calendar = Calendar.getInstance();
        return calendar.getTime();
    }

    public static Date getNextMonth() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH,1);
        return calendar.getTime();
    }

    public static Date getNextMonthByDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH,1);
        return calendar.getTime();
    }

    public static Date getFirstDateFromCurrMonth(){
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH,0);
        calendar.set(Calendar.DAY_OF_MONTH,1);
        return calendar.getTime();
    }

    public static Date getFirstDateFromCustomMonth(Date date){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH,0);
        calendar.set(Calendar.DAY_OF_MONTH,1);
        return calendar.getTime();
    }

    public static Date getLastDateFromCurrMonth(){
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH,1);
        calendar.set(Calendar.DAY_OF_MONTH,0);
        return calendar.getTime();
    }

    public static Date getLastDateFromCustomMonth(Date date){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH,1);
        calendar.set(Calendar.DAY_OF_MONTH,0);
        return calendar.getTime();
    }

    /**
     * 取得当月天数
     * */
    public static int getCurrentMonthLastDay(){
        Calendar a = Calendar.getInstance();
        a.set(Calendar.DATE, 1);//把日期设置为当月第一天
        a.roll(Calendar.DATE, -1);//日期回滚一天，也就是最后一天
        int maxDate = a.get(Calendar.DATE);
        return maxDate;
    }

    /**
     * 得到指定月的天数
     * */
    public static int getMonthLastDay(Date date){
        Calendar a = Calendar.getInstance();
        a.setTime(date);
        a.set(Calendar.DATE, 1);//把日期设置为当月第一天
        a.roll(Calendar.DATE, -1);//日期回滚一天，也就是最后一天
        int maxDate = a.get(Calendar.DATE);
        return maxDate;
    }

    /**
     * 获取周几(上周/下周)的时间
     * @param week 周几 	Calendar.MONDAY Calendar.FRIDAY...
     * @param weekNum 几周 	1[1周] 2[2周]...
     * @return
     */
    private static Date getLastOrNextWeek(int week,int weekNum) {
        GregorianCalendar currentDate = new GregorianCalendar();
        currentDate.setFirstDayOfWeek(Calendar.MONDAY);
        currentDate.set(GregorianCalendar.DAY_OF_WEEK, week);  // 设置星期为周week
        currentDate.add(GregorianCalendar.WEEK_OF_YEAR, weekNum); // 加/减weekNum周
        Date nextWeek = currentDate.getTime();
        return nextWeek;
    }

    public static Date getLastWeek(int weekNum,int nextNum) {
        return getLastOrNextWeek(weekNum,-nextNum);
    }

    public static Date getCurrWeek(int weekNum) {
        return getLastOrNextWeek(weekNum,0);
    }

    public static Date getNextWeek(int weekNum,int nextNum) {
        return getLastOrNextWeek(weekNum,nextNum);
    }

    /**
     * 获取上周五时间
     */
    private static Date getLastFriday() {
        return getLastWeek(Calendar.FRIDAY,1);
    }

    public static Date getLastFridayStart(){
        Date fridayStart = getLastFriday();
        return dayStart(fridayStart);
    }

    public static Date getLastFridayEnd(){
        Date fridayEnd = getLastFriday();
        return dayEnd(fridayEnd);
    }

    /**
     * 获得下周五时间
     */
    private static Date getNextFriday() {
        return getNextWeek(Calendar.FRIDAY,1);
    }

    public static Date getNextFridayStart() {
        Date fridayStart = getNextFriday();
        return dayStart(fridayStart);
    }

    public static Date getNextFridayEnd() {
        Date nextFridayEnd = getNextFriday();
        return dayEnd(nextFridayEnd);
    }

    /**
     * 获取本周五时间
     */
    public static Date getCurrFriday(){
        return getCurrWeek(Calendar.FRIDAY);
    }

    public static Date getCurrFridayStart(){
        Date date = getCurrFriday();
        return dayStart(date);
    }
    public static Date getCurrFridayEnd(){
        Date date = getCurrFriday();
        return dayEnd(date);
    }

    /**
     * 计算两个日期之间相差的天数
     * @param smdate 较小的时间
     * @param bdate  较大的时间
     * @return 相差天数
     * @throws ParseException
     */
    public static int daysBetween(Date smdate,Date bdate) throws ParseException{
        SimpleDateFormat sdf=new SimpleDateFormat("yyyy-MM-dd");
        smdate=sdf.parse(sdf.format(smdate));
        bdate=sdf.parse(sdf.format(bdate));
        Calendar cal = Calendar.getInstance();
        cal.setTime(smdate);
        long time1 = cal.getTimeInMillis();
        cal.setTime(bdate);
        long time2 = cal.getTimeInMillis();
        long betweenDays=(time2-time1)/(1000*3600*24);
        return Integer.parseInt(String.valueOf(betweenDays));
    }

    /**
     * 判断当前时间是否在一个区间内(包含开始时间和结束时间)
     *
     * @param strDateBegin
     *            开始时间 如:16:00:01
     * @param strDateEnd
     *            结束时间 如:23:59:59
     * @return
     * @throws IllegalArgumentException
     */
    public static boolean isInTime(String strDateBegin, String strDateEnd){
        // 截取开始时间时分秒
        int strDateBeginH = Integer.parseInt(strDateBegin.substring(0, 2));
        int strDateBeginM = Integer.parseInt(strDateBegin.substring(3, 5));
        int strDateBeginS = Integer.parseInt(strDateBegin.substring(6, 8));
        // 截取结束时间时分秒
        int strDateEndH = Integer.parseInt(strDateEnd.substring(0, 2));
        int strDateEndM = Integer.parseInt(strDateEnd.substring(3, 5));
        int strDateEndS = Integer.parseInt(strDateEnd.substring(6, 8));

        Calendar calendarStart = Calendar.getInstance();
        calendarStart.setTime(Calendar.getInstance().getTime());
        calendarStart.set(Calendar.HOUR_OF_DAY,strDateBeginH);
        calendarStart.set(Calendar.MINUTE,strDateBeginM);
        calendarStart.set(Calendar.SECOND,strDateBeginS);

        Calendar calendarEnd = Calendar.getInstance();
        calendarEnd.setTime(Calendar.getInstance().getTime());
        calendarEnd.set(Calendar.HOUR_OF_DAY,strDateEndH);
        calendarEnd.set(Calendar.MINUTE,strDateEndM);
        calendarEnd.set(Calendar.SECOND,strDateEndS);

        if(Calendar.getInstance().getTimeInMillis() >= calendarStart.getTimeInMillis()
                && Calendar.getInstance().getTimeInMillis()< calendarEnd.getTimeInMillis()){

            return true;
        }

        return false;
    }

    /**
     * 有效期
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean termOfValidity(Date startDate ,Date endDate){
        if(null == startDate || null == endDate){
            return false;
        }
        Calendar instance = Calendar.getInstance();
        Long nowTime = instance.getTime().getTime();
        if(nowTime>startDate.getTime() && nowTime<endDate.getTime()){
            return true;
        }
        return false;
    }


    public static Boolean isInTime(Date startDate,Date endDate ,Date date){
        long startTimes =0L;
        long endTime = 0L;
        long sourceTime = new Date().getTime();
        if(null != startDate){
            startTimes = startDate.getTime();
        }
        if(null != endDate){
            endTime = endDate.getTime();
        }
        if(null != date){
            sourceTime = date.getTime();
        }
        if(startTimes<=sourceTime && endTime >= sourceTime){
            return true;
        }
        return false;
    }

    /**
     * 获得指定日期的前一天
     *
     * @param specifiedDay
     * @return
     * @throws Exception
     */
    public static Date getSpecifiedDayBefore(String specifiedDay) {
        Calendar c = Calendar.getInstance();
        Date date = null;
        try {
            date = new SimpleDateFormat(DEFAULT_DATETIME_FORMAT).parse(specifiedDay);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        c.setTime(date);
        int day = c.get(Calendar.DATE);
        c.set(Calendar.DATE, day - 1);

        String dayBefore = new SimpleDateFormat(DEFAULT_DATETIME_FORMAT).format(c
                .getTime());
        Date beforeDay = parse(dayBefore, DEFAULT_DATETIME_FORMAT);
        return beforeDay;
    }

    /**
     * 通过时间秒毫秒数判断两个时间的间隔
     * @param date1
     * @param date2
     * @return
     */
    public static int differentDaysByMillisecond(Date date1,Date date2)
    {
        int days = (int) ((date2.getTime() - date1.getTime()) / (1000*3600*24));
        return days;
    }

    /**
     * @param :请求接口 http://api.goseek.cn/
     * @param currDate :日期参数yyyyMMdd
     * @return 返回结果 {"code":10000,"data":0}  0:工作日  1：星期天 2：节假日
     * 其它链接：http://tool.bitefu.net/jiari/vip.php
     */
    public static String fetchHoliday(String currDate) {
        BufferedReader reader = null;
        String result = null;
        JSONObject jsonObjectResult = null;
        StringBuffer sbf = new StringBuffer();
        String httpUrl = "http://tool.bitefu.net/jiari/?d=" + currDate;
        try {
            URL url = new URL(httpUrl);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("GET");
            connection.connect();
            InputStream is = connection.getInputStream();
            reader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
            String strRead = null;
            while ((strRead = reader.readLine()) != null) {
                sbf.append(strRead);
//                sbf.append("\r\n");
            }
            result = sbf.toString();
//            jsonObjectResult = JSONObject.parseObject(result);//转为JSONObject对象
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (reader != null){
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return result;
    }

    /**
     *
     * @param currDate
     * @return true:节假日 false：工作日
     */
    public static boolean checkHoliday(String currDate) {
        String date = fetchHoliday(currDate);
//        String date = jsonObject.getString("data");
        return checkHolidayByType(date);
    }

    public static boolean checkHolidayByType(String dateType){
        if (EmptyUtil.isNotEmpty(dateType) && (dateType.equals("1")|| dateType.equals("2"))){
            return true;
        }
        return false;
    }

    /**
     * 获取法定日期类型
     * @param currDate
     * @return 0工作日，1星期天，2：节假日
     */
    public static int fetchStatutoryHolidayType(String currDate) {
//        JSONObject jsonObject = fetchHoliday(currDate);
        String date = fetchHoliday(currDate);
        if (EmptyUtil.isNotEmpty(date)){
            try{
                return Integer.parseInt(date);
            }catch (NumberFormatException e){
            }
        }
        return 0;
    }

    /**
     * 获取1年中每天的 格式化日期值
     * @param date
     * @return
     */
    public static List<String> fetchDayOfYear(Date date){
        List<String> daysOfYearList = new ArrayList<>();
        int days = getDayOfYearByYear(date);
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        for (int i=1;i<=days;i++){
            c.set(Calendar.DAY_OF_YEAR,i);
            daysOfYearList.add(DateUtil.format(c.getTime(),DateUtil.DEFAULTDATEFORMAT));
        }
        return daysOfYearList;
    }

    // 获取当年的天数
    public static int getDayOfYear() {
        Calendar c = Calendar.getInstance();
        return c.getActualMaximum(Calendar.DAY_OF_YEAR);
    }

    // 获取指定 日期所属年份的天数
    public static int getDayOfYearByYear(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.getActualMaximum(Calendar.DAY_OF_YEAR);
    }

    // 统计从当天至指定日期内和天数，除去节假日及星期
    public static int countCurDateToHolidayDays(Date minDate,Date currDate,Map<String,Boolean> dayHolidayMap){
        // 统计节假日天数
        int holidayDays = 0;
        Date addDate = null;
        Boolean isHoliday = false;
        try{
            int intervalDays = DateUtil.daysBetween(minDate,currDate);
            if (intervalDays>=0){
                for (int i=0;i<=intervalDays;i++){
                    addDate = DateUtil.addDay(currDate,-i);
                    isHoliday = dayHolidayMap.get(DateUtil.format(addDate,DateUtil.DEFAULTDATEFORMAT));
                    if (isHoliday == null){
                        isHoliday = checkHolidayByDate(addDate);
                        dayHolidayMap.put(DateUtil.format(addDate,DateUtil.DEFAULTDATEFORMAT),isHoliday);
                    }
                    if (isHoliday){
                        holidayDays += 1;
                    }
                }
            }
        }catch (Exception e){
            e.printStackTrace();
        }
        return holidayDays;
    }

    private static boolean checkHolidayByDate(Date date){
        return DateUtil.checkHoliday(DateUtil.format(date,DateUtil.DEFAULTDATEFORMAT));
    }

    /**
     *
     * @param currDate
     * @return 0:工作日  1：星期天 2：节假日
     */
    public static Integer fetchHolidayNum(String currDate) {
//        JSONObject jsonObject = fetchHoliday(currDate);
        String date = fetchHoliday(currDate);
        if (EmptyUtil.isNotEmpty(date)){
            return Integer.parseInt(date);
        }
        return 0;
    }

    public static void main(String[] args) {
        try {
			/*if("2017-08-14".compareTo("2017-08-15")>0){
				System.out.println(true);
			}else{
				System.out.println(false);
			}*/
            //System.out.println(DateUtil.format(getNextMonth(),YEAR_MONTH_FORMAT));
            /*System.out.println(DateUtil.format(new Date(117,11,12),DEFAULT_DATE_FORMAT));
            System.out.println(DateUtil.format(getNextMonthByDate(new Date(117,11,12)),DEFAULT_DATETIME_FORMAT));
            System.out.println(DateUtil.format(getFirstDateFromCurrMonth(),DEFAULT_DATETIME_FORMAT));
            System.out.println(DateUtil.format(getLastDateFromCurrMonth(),DEFAULT_DATETIME_FORMAT));
            System.out.println(DateUtil.format(getFirstDateFromCustomMonth(new Date(117,11,12)),DEFAULT_DATETIME_FORMAT));
            System.out.println(DateUtil.format(getLastDateFromCustomMonth(new Date(117,11,12)),DEFAULT_DATETIME_FORMAT));
            System.out.println(getMonthLastDay(new Date()));*/
            // System.out.println(format(getCurrDateByDateStr("2018-04-16"),DEFAULT_DATE_FORMAT));

            //boolean a = DateUtil.checkHoliday("20180101");
            //System.out.println("节假日："+a);

            //Map<String,Boolean> dayHolidayMap = new HashMap<>();
            //System.out.println(countCurDateToHolidayDays(DateUtil.addDay(new Date(),-4),new Date(),dayHolidayMap));

            System.out.println(getDayOfYearByYear(new Date()));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    //获取指定月份的天数
    public static int getDaysByYearMonth(int year, int month) {
        Calendar a = Calendar.getInstance();
        a.set(Calendar.YEAR, year);
        a.set(Calendar.MONTH, month - 1);
        a.set(Calendar.DATE, 1);
        a.roll(Calendar.DATE, -1);
        int maxDate = a.get(Calendar.DATE);
        return maxDate;
    }

    public static String getFirstOrLastDayOfMonth (String date,Integer firstOrLast) {
        String[] dateArray = date.split("-");
        Integer year = Integer.parseInt(dateArray[0]);
        Integer month = Integer.parseInt(dateArray[1]);
        Calendar cal = Calendar.getInstance();
        //设置年份
        cal.set(Calendar.YEAR,year);
        //设置月份
        cal.set(Calendar.MONTH, month-1);
        if (firstOrLast == 0) {
            //获取某月最小天数
            int firstDay = cal.getActualMinimum(Calendar.DAY_OF_MONTH);
            //设置日历中月份的最小天数
            cal.set(Calendar.DAY_OF_MONTH, firstDay);
        } else {
            //获取某月最大天数
            int lastDay = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
            //设置日历中月份的最大天数
            cal.set(Calendar.DAY_OF_MONTH, lastDay);
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String lastDayOfMonth = sdf.format(cal.getTime());
        return lastDayOfMonth;
    }

    public static final Date parseUnixTimestamp(Long unixTimestamp) {
        SimpleDateFormat sdf = new SimpleDateFormat(FORMAT_YYYY_MM_DD_HH_MM_SS);
        Date date = new Date(unixTimestamp * 1000l);
        String formattedDate = sdf.format(date);
        return parse(formattedDate, FORMAT_YYYY_MM_DD_HH_MM_SS);
    }
}
