package com.thebeastshop.common.aop;

import com.alibaba.fastjson.serializer.SerializerFeature;
import com.beast.clog.agent.TagBuilder;
import com.beast.clog.agent.metircs.IMetrics;
import com.beast.clog.agent.metircs.MetricsManager;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.thebeastshop.common.Page;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.exception.BusinessException;
import com.thebeastshop.common.prop.PropConstants;
import com.thebeastshop.common.spring.SpringAware;
import com.thebeastshop.common.utils.Base64Utils;
import com.thebeastshop.common.utils.GZIPHelper;

import jodd.util.StringUtil;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSON;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;

/**
 * <p>Title: thebeastshop</p>
 * <p>Copyright: Copyright (c) 2016</p>
 * @author Bryan Zhang
 * @date 2017-1-19
 * @description 日志
 */
public class ServiceAop {
	
	private final Logger log = LoggerFactory.getLogger(getClass());
	
	private int argLength = 200;
	
	private boolean needCompress;

	private final IMetrics metrics = MetricsManager.getMetrics();

    private final static String METRICS_SERVICE_ALL = "service.all";

	private InetAddress addr;
	private String hostName;
	{
		try {
			addr = InetAddress.getLocalHost();
			hostName = addr.getHostName().toString();
		} catch (UnknownHostException e) {
			e.printStackTrace();
		}
	}

	private void metricServiceAll(String appId, String className, String methodName, boolean success, Date startTime) {
		try {
			Date endTime = new Date();
			long duration = endTime.getTime() - startTime.getTime();
			String metricsName = appId.replaceAll("-", ".") + '.' + METRICS_SERVICE_ALL;
			String serviceName = className + '.' + methodName;
			TagBuilder tagBuilder = TagBuilder.create()
					.append("success", success)
					.append("service", serviceName);
			if (StringUtil.isNotEmpty(hostName)) {
				tagBuilder.append("host", hostName);
			}
			metrics.record(metricsName, duration, tagBuilder.build());
		} catch (Throwable t) {
		}
    }
	
	public Object doAround(ProceedingJoinPoint pJoinPoint){
		try{
		    final String appId = PropConstants.getAppId();
			Class clazz = pJoinPoint.getTarget().getClass();
			String className = clazz.getSimpleName();
			Signature signature = pJoinPoint.getSignature();
			MethodSignature methodSignature = (MethodSignature)signature;
			Method method = methodSignature.getMethod();
	        String methodName = method.getName();
			log.info("方法[{}.{}]开始调用",className,methodName);
			Object arg = null;
			StringBuilder argsBuilder = new StringBuilder("(");
			int argsLen = pJoinPoint.getArgs().length;
			try {
				for (int i = 0; i < argsLen; i++) {
					arg = pJoinPoint.getArgs()[i];
					int argIndex = i + 1;
					String argValue = getArgValue(arg);
					argsBuilder.append("参数" + argIndex + ": " + argValue);
					if (i < argsLen - 1) {
						argsBuilder.append(", ");
					}
					log.info("方法[{}.{}]参数{}:{}", className, methodName, argIndex, argValue);
				}
				argsBuilder.append(")");
				Cat.logEvent("PigeonService.args",
						className + "." + methodName,
						Message.SUCCESS,
						argsBuilder.toString());
			} catch (Exception e) {
				log.error("" + e);
			}
			long start = System.currentTimeMillis();
			Object result = null;
			Date startTime = new Date();
			try {
				result = pJoinPoint.proceed();
				metricServiceAll(appId, className, methodName, true, startTime);
			} catch (Throwable t) {
				if (t instanceof BusinessException) {
					log.info(t.getMessage());
				}
				else {
					log.error("系统发生异常: ", t);
				}
				metricServiceAll(appId, className, methodName, false, startTime);
				Class retType = method.getReturnType();
				if (ServiceResp.class.isAssignableFrom(retType)) {
					result = ServiceResp.newInstanceFail(t);
					try {
						// 回滚当前事务
						SpringAware.rollBack();
					} catch (Throwable t2) {
					}
				}
				else {
					throw t;
				}
			}
			// 添加Service元信息
			if (result != null && result instanceof ServiceResp) {
				ServiceResp resp = (ServiceResp) result;
				resp.setAppId(appId);
			}
			long end = System.currentTimeMillis();
			
			log.info("方法[{}.{}]结束调用,总共耗时{}毫秒", className, methodName, end - start);
			return result;
		}catch(Throwable t){
			if (t instanceof RuntimeException) {
				throw (RuntimeException) t;
			}
			throw new RuntimeException(t);
		}
	}


	private String getArgValue(Object arg) {
		String argValue;
		if (arg == null) {
			return "null";
		}
		if (arg instanceof CharSequence) {
			return "\"" + arg.toString() + "\"";
		}
		if (int.class.isAssignableFrom(arg.getClass()) ||
				long.class.isAssignableFrom(arg.getClass()) ||
				short.class.isAssignableFrom(arg.getClass()) ||
				byte.class.isAssignableFrom(arg.getClass()) ||
				boolean.class.isAssignableFrom(arg.getClass())) {
			return String.valueOf(arg);
		}
		try {
			argValue = JSON.toJSONString(arg, SerializerFeature.DisableCircularReferenceDetect);
			
			if(argLength == -1 && needCompress) {
				argValue = GZIPHelper.zipAndSerialize(argValue);
			}else if(argLength != -1 && argValue.length() > argLength){
				argValue = argValue.substring(0,argLength) + " ...";
			}
		} catch (Exception ex) {
			try {
				argValue = arg.getClass().getSimpleName() + " " + String.valueOf(arg);
			} catch (Exception ex2) {
				argValue = "null";
			}
		}
		return argValue;
	}


	public int getArgLength() {
		return argLength;
	}


	public void setArgLength(int argLength) {
		this.argLength = argLength;
	}


	public boolean isNeedCompress() {
		return needCompress;
	}


	public void setNeedCompress(boolean needCompress) {
		this.needCompress = needCompress;
	}

}
