package com.thebeastshop.common.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.apache.commons.io.output.FileWriterWithEncoding;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.thebeastshop.common.enums.FileTypeEnum;

/**
 * 读取文件工具类
 * 
 * @author everywhere.z
 * @date 2015-3-5
 * @description
 */
public class FileUtil {

	private final static Logger logger = LoggerFactory.getLogger(FileUtil.class);

	public final static String LINE_SEPARATOR = System.getProperty("line.separator");

	@SuppressWarnings("serial")
	private final static Map<String, FileTypeEnum> FILE_TYPE = new HashMap<String, FileTypeEnum>() {
		{
			// 常用文件类型的魔数
			put("FFD8FF", FileTypeEnum.JPG);
			put("89504E47", FileTypeEnum.PNG);
			put("47494638", FileTypeEnum.GIF);
			put("5249464", FileTypeEnum.WAV);
			put("4944330", FileTypeEnum.MP3);
		}
	};

	public static boolean createDirectory(File file) {
		if (file.exists()) {
			return true;
		}
		return file.mkdirs();
	}

	public static boolean createDirectory(String dirname) {
		return createDirectory(new File(dirname));
	}

	public static String readFile2String(String filename) throws IOException {
		return readFile2String(new File(filename));
	}

	public static String readFile2String(File file) throws IOException {
		if ((file == null) || !file.exists() || file.isDirectory()) {
			return null;
		}
		return readInputStream2String(new FileInputStream(file));
	}

	public static String readInputStream2String(InputStream is) throws IOException {
		return readInputStream2String(is, "UTF-8");
	}

	public static String readInputStream2String(InputStream is, String charset) throws IOException {
		BufferedReader br = null;
		StringBuilder sb = new StringBuilder();
		try {
			br = new BufferedReader(new InputStreamReader(is, charset));
			for (String str = null; (str = br.readLine()) != null;) {
				sb.append(str).append(LINE_SEPARATOR);
			}
		} finally {
			closeIO(br);
		}
		return sb.toString().trim();
	}

	public static List<String> readFile2List(String filename) throws IOException {
		return readFile2List(new File(filename));
	}

	public static List<String> readFile2List(File file) throws IOException {
		if ((file == null) || !file.exists() || file.isDirectory()) {
			return null;
		}
		BufferedReader br = null;
		List<String> list = new ArrayList<String>();
		try {
			br = new BufferedReader(new FileReader(file));
			for (String str = null; (str = br.readLine()) != null;) {
				list.add(str);
			}
		} finally {
			closeIO(br);
		}
		return list;
	}

	public static void writeString2File(String str, String filename) throws IOException {
		writeString2File(str, new File(filename));
	}

	public static void writeString2File(String str, File file) throws IOException {
		BufferedWriter bw = null;
		try {
			bw = new BufferedWriter(new FileWriterWithEncoding(file, "UTF-8"));
			bw.write(str);
			bw.flush();
		} finally {
			closeIO(bw);
		}
	}

	public static void writeList2File(List<String> list, String filename) throws IOException {
		writeList2File(list, new File(filename), LINE_SEPARATOR);
	}

	public static void writeList2File(List<String> list, File file) throws IOException {
		writeList2File(list, file, LINE_SEPARATOR);
	}

	public static void writeList2File(List<String> list, String filename, String lineSeparator) throws IOException {
		writeList2File(list, new File(filename), lineSeparator);
	}

	public static void writeList2File(List<String> list, File file, String lineSeparator) throws IOException {
		StringBuffer sb = new StringBuffer();
		for (int i = 0, k = list.size(); i < k; i++) {
			if (i > 0) {
				sb.append(lineSeparator);
			}
			sb.append(list.get(i));
		}
		writeString2File(sb.toString(), file);
	}

	public static void closeIO(Closeable io) throws IOException {
		if (io != null) {
			io.close();
		}
	}

	public static void zipFile(File inFile, File outFile) throws IOException {
		ZipOutputStream zos = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(outFile)));
		ZipEntry ze = new ZipEntry(inFile.getName());
		zos.putNextEntry(ze);

		byte[] buf = new byte[2048];
		int readLen = 0;
		InputStream is = new BufferedInputStream(new FileInputStream(inFile));
		while ((readLen = is.read(buf, 0, 2048)) != -1) {
			zos.write(buf, 0, readLen);
		}
		is.close();

		zos.close();
	}

	/**
	 * 获得指定文件的byte数组
	 */
	public static byte[] getBytes(String filePath) {
		byte[] buffer = null;
		try {
			File file = new File(filePath);
			FileInputStream fis = new FileInputStream(file);
			ByteArrayOutputStream bos = new ByteArrayOutputStream(1000);
			byte[] b = new byte[1000];
			int n;
			while ((n = fis.read(b)) != -1) {
				bos.write(b, 0, n);
			}
			fis.close();
			bos.close();
			buffer = bos.toByteArray();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return buffer;
	}

	/**
	 * 获取文件类型
	 * 
	 * @param filePath
	 * @return
	 */
	public static FileTypeEnum getFileType(String filePath) {
		return getFileType(new File(filePath));
	}

	/**
	 * 获取文件类型
	 * 
	 * @param file
	 * @return
	 */
	public static FileTypeEnum getFileType(File file) {
		FileTypeEnum fileType = null;
		try {
			fileType = getFileType(new FileInputStream(file));
		} catch (Exception e) {
			logger.error("获取文件类型异常: {}", e);
		}
		return fileType;
	}

	/**
	 * 获取文件类型
	 * 
	 * @param inputStream
	 * @return
	 */
	public static FileTypeEnum getFileType(InputStream inputStream) {
		FileTypeEnum fileType = null;
		// 数据流提取出字节数组
		try {
			byte[] b = new byte[28];
			inputStream.read(b);
			String hex = ByteUtil.bytes2Hex(b).toUpperCase();
			logger.info("文件魔数值: {}", hex);
			Set<String> keys = FILE_TYPE.keySet();
			for (String k : keys) {
				if (hex.startsWith(k)) {
					fileType = FILE_TYPE.get(k);
					break;
				}
			}
		} catch (Exception e) {
			logger.error("获取文件类型异常: {}", e);
		}
		return fileType;
	}

}
